package org.eclipse.expi;

import java.io.IOException;
import java.net.JarURLConnection;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.jar.Attributes;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.MessageBox;

public abstract class AbstractProgramLauncher implements SelectionListener {
    public static class LaunchFailedException extends Exception {
        /**
         *
         */
        private static final long serialVersionUID = 1L;

        public LaunchFailedException(final String message, final Throwable reason) {
            super(message, reason);
        }

        public LaunchFailedException(final String message) {
            super(message);
        }
    }

    /** Control widget that may trigger the program launcher. */
    private final Control control;

    protected abstract void launch() throws LaunchFailedException;

    /** Path to the jar file containing the program to be launched by this launcher. */
    private String program = null;

    public AbstractProgramLauncher(final Control control) {
        control.setEnabled(false);
        control.setVisible(false);
        this.control = control;
    }

    private void setEnable(final boolean enable) {
        control.setVisible(enable);
        control.setEnabled(enable);
    }

    /**
     * Sets the program to be launched by this launcher.
     *
     * @param program
     *            The path to the jar file containing the program. Pass null to disable the
     *            launcher.
     */
    public void setProgram(final String program) {
        this.program = program == null ? null : program.trim();
        setEnable(program != null && !"".equals(this.program));
    }

    @Override
    public void widgetDefaultSelected(final SelectionEvent arg0) {
    }

    @Override
    public void widgetSelected(final SelectionEvent arg0) {
        try {
            FileLogger.getLogger("interaction").log("RUN " + program);
            launch();
        } catch (final LaunchFailedException e) {
            FileLogger.getLogger("error").log(e);
            final MessageBox mb = new MessageBox(Display.getCurrent().getActiveShell(),
                    SWT.ICON_ERROR);
            mb.setText("Fail! " + e.getMessage());
            final Throwable cause = e.getCause();
            if (cause != null) {
                mb.setMessage(cause.getMessage());
            }
            mb.open();
        }
    }

    protected URL getProgramUrl() throws LaunchFailedException {
        final String workSpacePath = Util.getWorkspacePath();
        final StringBuilder urlBuilder = new StringBuilder(workSpacePath).append('/').append(
                program);
        final URL url;
        try {
            final String urlString = urlBuilder.toString();
            final URI uri = new URI("file", urlString, null);
            url = uri.toURL();
        } catch (final MalformedURLException e) {
            throw new LaunchFailedException("The filename " + program
                    + " does not point to a valid URL. Check tasks.xml", e);
        } catch (final URISyntaxException e) {
            throw new LaunchFailedException("The filename " + program
                    + " does not point to a valid URL. Check tasks.xml", e);
        }
        return url;
    }

    protected String getMainClass(final URL url) throws LaunchFailedException {
        // Open jar file.
        final JarURLConnection jar;
        try {
            final URL jarUrl = new URL("jar", "", url + "!/");
            jar = (JarURLConnection) jarUrl.openConnection();
        } catch (final IOException e) {
            throw new LaunchFailedException(program + " can't be started.", e);
        }

        // Get main class from jar file.
        final Attributes attributes;
        try {
            attributes = jar.getMainAttributes();
        } catch (final IOException e) {
            throw new LaunchFailedException("Can't read attributes from " + program + ".", e);
        }
        assert attributes != null;
        final String mainClassName = attributes.getValue(Attributes.Name.MAIN_CLASS);
        if (mainClassName == null) {
            throw new LaunchFailedException(program + " has no attribute "
                    + Attributes.Name.MAIN_CLASS);
        }
        return mainClassName;
    }

    public String getProgram() {
        return program;
    }
}