package org.eclipse.expi;

import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.jobs.IJobChangeEvent;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.core.runtime.jobs.JobChangeAdapter;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchListener;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorReference;
import org.eclipse.ui.IPageListener;
import org.eclipse.ui.IPartListener2;
import org.eclipse.ui.IPropertyListener;
import org.eclipse.ui.IViewReference;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPartReference;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;

/**
 * Logs what's been going on on the workspace.
 *
 * @author beccs
 */
public final class ActionMonitor implements IPageListener, IPartListener2, Listener,
        IPropertyListener {
    private final class LaunchListener implements ILaunchListener {
        @Override
        public void launchRemoved(final ILaunch l) {
            // ignore
        }

        @Override
        public void launchChanged(final ILaunch l) {
            /*
             * Ignore changes if no launch jobs are running. We need this since debug launches cause
             * two changes of the launch objects.
             */
            if (launchJobs-- > 0) {
                final ILaunchConfiguration lc = l.getLaunchConfiguration();
                try {
                    final Object mainObj = lc.getAttributes().get(
                            "org.eclipse.jdt.launching.MAIN_TYPE");
                    final String main = mainObj != null ? mainObj.toString() : "";
                    FileLogger.getLogger("interaction").log(
                            "LAUNCH " + l.getLaunchMode() + " " + lc.getName() + " " + main);
                } catch (final CoreException e) {
                    return;
                }
            }
        }

        @Override
        public void launchAdded(final ILaunch l) {
            // ignore
        }
    }

    private static ActionMonitor instance;

    /** Number of started launch jobs. */
    private int                  launchJobs = 0;

    public static ActionMonitor getInstance() {
        if (instance == null) {
            instance = new ActionMonitor();
        }
        return instance;
    }

    private ActionMonitor() {
        log("--- start ---");
        final IWorkbenchWindow wwin = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
        wwin.addPageListener(this);
        wwin.getPartService().addPartListener(this);
        final IWorkbenchPage[] pages = wwin.getPages();
        for (final IWorkbenchPage p : pages) {
            final IEditorPart editorPart = p.getActiveEditor();
            if (editorPart != null) {
                editorPart.addPropertyListener(this);
            }
            log("found page: " + format(p));
            for (final IEditorReference eref : p.getEditorReferences()) {
                try {
                    log("Page name: " + eref.getName());
                    final Shell s = eref.getEditor(false).getSite().getShell();
                    s.addListener(SWT.MouseMove, this);

                } catch (final NullPointerException e) {
                    continue;
                }
            }
            for (final IViewReference vref : p.getViewReferences()) {
                try {
                    final Shell s = vref.getView(false).getSite().getShell();
                    s.addListener(SWT.MouseMove, this);
                } catch (final NullPointerException e) {
                    continue;
                }
            }
        }

        // listen to new jobs and increment the counter of launched jobs.
        Job.getJobManager().addJobChangeListener(new JobChangeAdapter() {
            @Override
            public void aboutToRun(final IJobChangeEvent event) {
                if (event.getJob().getName().startsWith("Launching")) {
                    ++launchJobs;
                }
            }
        });

        DebugPlugin.getDefault().getLaunchManager().addLaunchListener(new LaunchListener());
    }

    private String format(final IWorkbenchPage p) {
        return p + " / " + p.getLabel();
    }

    private String content(final IWorkbenchPartReference p) {
        return "id=" + p.getId() + p.getContentDescription();
    }

    private String format(final IWorkbenchPartReference p) {
        return "id=" + p.getId() + ", partname=" + p.getPartName() + ", title=" + p.getTitle()
                + ", content=" + p.getTitleToolTip() + content(p);
    }

    private String format(final IEditorPart p) {
        return "id=" + p.getTitle() + " + content=" + p.getTitleToolTip() + " saved: "
                + p.isDirty();
    }

    // IPageListener

    @Override
    public void pageActivated(final IWorkbenchPage page) {
        log("page activated: " + format(page));
    }

    @Override
    public void pageClosed(final IWorkbenchPage page) {
        log("page closed: " + format(page));
    }

    @Override
    public void pageOpened(final IWorkbenchPage page) {
        log("page opened: " + format(page));
    }

    // IPartListener2

    @Override
    public void partActivated(final IWorkbenchPartReference partRef) {
        log("part activated: " + format(partRef));
    }

    @Override
    public void partBroughtToTop(final IWorkbenchPartReference partRef) {
        log("part to top: " + format(partRef));
    }

    @Override
    public void partClosed(final IWorkbenchPartReference partRef) {
        log("part closed: " + format(partRef));
    }

    @Override
    public void partDeactivated(final IWorkbenchPartReference partRef) {
        log("part deactivated: " + format(partRef));
    }

    @Override
    public void partHidden(final IWorkbenchPartReference partRef) {
        log("part hidden: " + format(partRef));
    }

    @Override
    public void partInputChanged(final IWorkbenchPartReference partRef) {
        log("part input changed: " + format(partRef));
    }

    @Override
    public void partOpened(final IWorkbenchPartReference partRef) {
        log("part opened: " + format(partRef));
    }

    @Override
    public void partVisible(final IWorkbenchPartReference partRef) {
        log("part visible: " + format(partRef));
    }

    /**
     * Logs key events to keys.log
     *
     * @param event
     *            the event.
     */
    @Override
    public void handleEvent(final Event event) {
        if (event.type == SWT.KeyUp || event.type == SWT.KeyDown) {
            String t = "UP";
            if (event.type == SWT.KeyDown) {
                t = "DOWN";
            }
            FileLogger.getLogger("keys").log(
                    String.format("%s %d %d", t, event.keyCode, event.stateMask));
        }
    }

    public void log(final String msg) {
        FileLogger.getLogger("workspace").log(msg);
    }

    public String getProjectPath() {
        return ResourcesPlugin.getWorkspace().getRoot().getLocation().toString();
    }

    @Override
    public void propertyChanged(final Object source, final int propId) {
        if (propId == IEditorPart.PROP_DIRTY) {
            log("part edit: " + source + format((IEditorPart) source));
        }

    }
}
