package org.eclipse.expi;

import java.io.FileDescriptor;
import java.net.InetAddress;
import java.security.Permission;

/**
 * SecurityManager which takes another SecurityManager which it delegates every check to. The
 * exception is that it prevents the program from exiting the JVM *once*.
 *
 * After it denied exiting it installs the delegate as the SecurityManager for the VM.
 *
 * @author Felix Wenk
 */
public class JarSecurityManager extends SecurityManager {
    private static class ExitJarException extends SecurityException {
        /**
         *
         */
        private static final long serialVersionUID = 1L;

        public ExitJarException() {
            super("System.exit called from the embedded program."
                    + "Don't worry about this exception.");
        }
    }

    private final SecurityManager original;

    public JarSecurityManager(final SecurityManager delegate) {
        this.original = delegate;
    }

    @Override
    public void checkExit(final int arg0) {
        // Restore old security manager.
        System.setSecurityManager(original);
        throw new ExitJarException();
    }

    @Override
    public void checkAccept(final String host, final int port) {
        if (original == null) {
            return;
        }
        original.checkAccept(host, port);
    }

    @Override
    public void checkAccess(final Thread t) {
        if (original == null) {
            return;
        }
        original.checkAccess(t);
    }

    @Override
    public void checkAccess(final ThreadGroup g) {
        if (original == null) {
            return;
        }
        original.checkAccess(g);
    }

    @Override
    public void checkAwtEventQueueAccess() {
        if (original == null) {
            return;
        }
        original.checkAwtEventQueueAccess();
    }

    @Override
    public void checkConnect(final String host, final int port, final Object context) {
        if (original == null) {
            return;
        }
        original.checkConnect(host, port, context);
    }

    @Override
    public void checkConnect(final String host, final int port) {
        if (original == null) {
            return;
        }
        original.checkConnect(host, port);
    }

    @Override
    public void checkCreateClassLoader() {
        if (original == null) {
            return;
        }
        original.checkCreateClassLoader();
    }

    @Override
    public void checkDelete(final String file) {
        if (original == null) {
            return;
        }
        original.checkDelete(file);
    }

    @Override
    public void checkExec(final String cmd) {
        if (original == null) {
            return;
        }
        original.checkExec(cmd);
    }

    @Override
    public void checkLink(final String lib) {
        if (original == null) {
            return;
        }
        original.checkLink(lib);
    }

    @Override
    public void checkListen(final int port) {
        if (original == null) {
            return;
        }
        original.checkListen(port);
    }

    @Override
    public void checkMemberAccess(final Class<?> clazz, final int which) {
        if (original == null) {
            return;
        }
        original.checkMemberAccess(clazz, which);
    }

    @SuppressWarnings("deprecation")
    @Override
    public void checkMulticast(final InetAddress maddr, final byte ttl) {
        if (original == null) {
            return;
        }
        original.checkMulticast(maddr, ttl);
    }

    @Override
    public void checkMulticast(final InetAddress maddr) {
        if (original == null) {
            return;
        }
        original.checkMulticast(maddr);
    }

    @Override
    public void checkPackageAccess(final String pkg) {
        if (original == null) {
            return;
        }
        original.checkPackageAccess(pkg);
    }

    @Override
    public void checkPackageDefinition(final String pkg) {
        if (original == null) {
            return;
        }
        original.checkPackageDefinition(pkg);
    }

    @Override
    public void checkPermission(final Permission perm, final Object context) {
        if (original == null) {
            return;
        }
        original.checkPermission(perm, context);
    }

    @Override
    public void checkPermission(final Permission perm) {
        if (original == null) {
            return;
        }
        original.checkPermission(perm);
    }

    @Override
    public void checkPrintJobAccess() {
        if (original == null) {
            return;
        }
        original.checkPrintJobAccess();
    }

    @Override
    public void checkPropertiesAccess() {
        if (original == null) {
            return;
        }
        original.checkPropertiesAccess();
    }

    @Override
    public void checkPropertyAccess(final String key) {
        if (original == null) {
            return;
        }
        original.checkPropertyAccess(key);
    }

    @Override
    public void checkRead(final FileDescriptor fd) {
        if (original == null) {
            return;
        }
        original.checkRead(fd);
    }

    @Override
    public void checkRead(final String file, final Object context) {
        if (original == null) {
            return;
        }
        original.checkRead(file, context);
    }

    @Override
    public void checkRead(final String file) {
        if (original == null) {
            return;
        }
        original.checkRead(file);
    }

    @Override
    public void checkSecurityAccess(final String target) {
        if (original == null) {
            return;
        }
        original.checkSecurityAccess(target);
    }

    @Override
    public void checkSetFactory() {
        if (original == null) {
            return;
        }
        original.checkSetFactory();
    }

    @Override
    public void checkSystemClipboardAccess() {
        if (original == null) {
            return;
        }
        original.checkSystemClipboardAccess();
    }

    @Override
    public boolean checkTopLevelWindow(final Object window) {
        if (original == null) {
            return true;
        }
        return original.checkTopLevelWindow(window);
    }

    @Override
    public void checkWrite(final FileDescriptor fd) {
        if (original == null) {
            return;
        }
        original.checkWrite(fd);
    }

    @Override
    public void checkWrite(final String file) {
        if (original == null) {
            return;
        }
        original.checkWrite(file);
    }
}
