package org.eclipse.expi;

import java.io.File;
import java.io.IOException;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * This class represents the current state in task processing. It determines which question or
 * information to show next.
 *
 * @author beccs
 */
public final class TaskProcessor {

    private static TaskProcessor instance;
    private List<Elem>           lNodes;
    private Iterator<Elem>       iter;
    private Elem                 current;
    private int                  nNumQuestions;
    private int                  nNumScreens;
    private int                  nCurrentQuestion = 0;
    private int                  nCurrentScreen   = 0;

    private TaskProcessor() {
    }

    public static TaskProcessor getInstance() {
        if (instance == null) {
            instance = new TaskProcessor();
            instance.initialize();
        }
        return instance;
    }

    public boolean more() {
        return iter != null && iter.hasNext();
    }

    public void next() {
        current = iter.next();
        if (current.open) {
            if (current.node.getNodeName().equals("question")) {
                nCurrentQuestion++;
                nCurrentScreen++;
            } else if (current.node.getNodeName().equals("info")) {
                nCurrentScreen++;
            } else if (current.node.getNodeName().equals("form")) {
                nCurrentScreen++;
            }
        }
    }

    public Node getNode() {
        return current.node;
    }

    public boolean isOpening() {
        return current.open;
    }

    public ProgressInfo getProgressInfo() {
        return new ProgressInfo(nCurrentQuestion, nCurrentScreen, nNumQuestions,
                nNumScreens);
    }

    private void initialize() {
        final DocumentBuilderFactory docBuilderFactory = DocumentBuilderFactory.newInstance();
        try {
            lNodes = new LinkedList<Elem>();
            final DocumentBuilder docBuilder = docBuilderFactory.newDocumentBuilder();
            final File file = new File(Util.getWorkspacePath() + "/tasks.xml");
            if (file.exists()) {
                final Document doc = docBuilder.parse(file);
                traverse(doc.getDocumentElement());
                iter = lNodes.iterator();
                current = iter.next();
                FileLogger.getLogger("interaction").log("Task file found: " + file);
            } else {
                FileLogger.getLogger("interaction").log("Task file not found: " + file);
            }
        } catch (ParserConfigurationException e) {
            FileLogger.getLogger("error").log(e);
        } catch (SAXException e) {
            FileLogger.getLogger("error").log(e);
        } catch (IOException e) {
            FileLogger.getLogger("error").log(e);
        }
    }

    private void traverse(final Node node) {
        lNodes.add(new Elem(node, true));
        if (node.getNodeName().equals("question")) {
            nNumQuestions++;
            nNumScreens++;
        } else if (node.getNodeName().equals("info")) {
            nNumScreens++;
        } else if (node.getNodeName().equals("form")) {
            nNumScreens++;
        }
        final NodeList nl = node.getChildNodes();
        for (int i = 0; i < nl.getLength(); i++) {
            traverse(nl.item(i));
        }
        lNodes.add(new Elem(node, false));
    }

    private static class Elem {
        private Node    node;
        private boolean open;

        Elem(final Node node, final boolean open) {
            this.node = node;
            this.open = open;
        }

        @Override
        public String toString() {
            if (open) {
                return "<" + node.getNodeName() + ">" + node.getNodeValue();
            } else {
                return "</" + node.getNodeName() + ">";
            }
        }
    }

    public static class ProgressInfo {
        private int numCurrentQuestion;
        private int numCurrentScreen;
        private int numNumQuestions;
        private int numNumScreens;

        public ProgressInfo(final int nCurrentQuestion, final int nCurrentScreen,
                final int nNumQuestions, final int nNumScreens) {
            this.numCurrentQuestion = nCurrentQuestion;
            this.numCurrentScreen = nCurrentScreen;
            this.numNumQuestions = nNumQuestions;
            this.numNumScreens = nNumScreens;
        }

        public boolean isFinished() {
            return numNumScreens == numCurrentScreen;
        }

        public int getNumCurrentQuestion() {
            return numCurrentQuestion;
        }

        public int getNumCurrentScreen() {
            return numCurrentScreen;
        }

        public int getNumNumQuestions() {
            return numNumQuestions;
        }

        public int getNumNumScreens() {
            return numNumScreens;
        }
    }
}
