package org.eclipse.expi;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URI;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

/**
 * Recursive zip tool.
 *
 * @author ojlr
 *
 */
public final class Zipper {
    /** The size of the buffer. */
    private static final int BUFFER_SIZE = 4096;
    /** The basepath that should be cut off from all files. */
    private final URI             basepath;
    /** The buffer that is used to read the files. */
    private final byte[]          buffer;
    /** The output stream of the zip file. */
    private final ZipOutputStream out;

    /**
     * Creates a new zipper.
     *
     * @param outFile
     *            The path of the zip file. Should end with .zip.
     * @param basepath
     *            The basepath of all elements of the zip file.
     * @throws FileNotFoundException
     *             If basepath was not found.
     */
    private Zipper(final File outFile, final File basepath) throws FileNotFoundException {
        this.basepath = basepath.toURI();
        buffer = new byte[BUFFER_SIZE];
        out = new ZipOutputStream(new FileOutputStream(outFile));
    }

    /**
     * Zips inFile to outFile.
     * 
     * @param outFile
     *            The destination file. Should end with .zip.
     * @param inFile
     *            The source file/directory.
     */
    public static void zip(final File outFile, final File inFile) {
        try {
            final Zipper zipper = new Zipper(outFile, inFile.getParentFile());
            zipper.zip(inFile);
            zipper.out.close();
        } catch (final FileNotFoundException e) {
            e.printStackTrace();
        } catch (final IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * Zips the content of currentFile.
     *
     * @param currentFile
     *            The file/directory that should be zipped.
     * @throws IOException
     *             If an error occures in the underling stream.
     */
    private void zip(final File currentFile) throws IOException {
        if (currentFile.isDirectory()) {
            for (final File f : currentFile.listFiles()) {
                zip(f);
            }
        } else {
            final FileInputStream in = new FileInputStream(currentFile);
            final String entryName = basepath.relativize(currentFile.toURI()).getPath();
            out.putNextEntry(new ZipEntry(entryName));
            int len;
            while ((len = in.read(buffer)) > 0) {
                out.write(buffer, 0, len);
            }
            out.closeEntry();
            in.close();
        }
    }
}
