package org.eclipse.expi.forms;

import java.util.Collection;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Dialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.wb.swt.SWTResourceManager;

// CHECKSTYLE:OFF
/**
 * Is used to display a composition of forms to ask the user some questions.
 *
 * @author ojlr
 *
 */
public class FormDialog extends Dialog {

    /** The default height of the shown dialog. */
    private static final int DEFAULT_HEIGHT = 600;
    /** The default width of the shown dialog. */
    private static final int DEFAULT_WIDTH = 700;
    /** The results of all forms. */
    private Collection<Answer> result;
    /** The shell objects that is used to display the dialog. */
    private Shell              shell;
    /** The headline of the questions. */
    private Group              grpName;
    /** The root form of the dialog. */
    private Form               form;
    /** The button that is used to "send" the form. */
    private Button             btnNext;
    /** The label that is used to display a message if some data of a form is invalid. */
    private Label              lblError;
    /** The error message that should be displayed in lblError. */
    private String             error;
    /** Used for scrolling. */
    private ScrolledComposite  scrolledComposite;
    /** Indicates if the dialog is modal. */
    private boolean            isModal = true;

    /**
     * Create the dialog.
     *
     * @param parent
     *            The parent composite.
     * @param style
     *            The swt style.
     * @param name
     *            The title of the dialog.
     * @param isModal
     *            Indicates if the dialog should be modal.
     */
    public FormDialog(final Composite parent, final int style, final String name,
            final boolean isModal) {
        super(parent.getShell(), SWT.NONE);
        this.isModal = isModal;
        setText(name);
        createContents();
    }

    public ScrolledComposite getFormComposite() {
        return scrolledComposite;
    }

    public void setForm(final Form f, final Control c) {
        assert f != null;
        this.form = f;
        c.pack();
        scrolledComposite.setContent(c);
        scrolledComposite.setMinSize(c.getSize());
    }

    public void setNextEnabled(final boolean b) {
        btnNext.setEnabled(b);
    }

    /**
     * Open the dialog.
     *
     * @return the result
     */
    public Collection<Answer> open() {
        shell.open();
        shell.layout();
        final Display display = getParent().getDisplay();
        while (!shell.isDisposed()) {
            if (!display.readAndDispatch()) {
                display.sleep();
            }
        }
        return result;
    }

    /**
     * Create contents of the dialog.
     */
    private void createContents() {
        shell = new Shell(getParent(), SWT.BORDER | SWT.RESIZE | SWT.TITLE
                | (isModal ? SWT.APPLICATION_MODAL : 0));
        shell.setSize(DEFAULT_WIDTH, DEFAULT_HEIGHT);
        shell.setText(getText());
        shell.setLayout(new GridLayout(2, false));

        shell.addListener(SWT.Traverse, new Listener() {
            @Override
            public void handleEvent(final Event e) {
                if (e.detail == SWT.TRAVERSE_ESCAPE) {
                    e.doit = false;
                }
            }
        });
        shell.addListener(SWT.Close, new Listener() {
            @Override
            public void handleEvent(final Event e) {
                e.doit = false;
            }
        });

        grpName = new Group(shell, SWT.NONE);
        grpName.setLayout(new FillLayout(SWT.HORIZONTAL));
        final GridData gd_grpName = new GridData(SWT.FILL, SWT.FILL, true, true, 2, 1);
        gd_grpName.widthHint = 135;
        gd_grpName.heightHint = 77;
        grpName.setLayoutData(gd_grpName);
        grpName.setText(this.getText());

        scrolledComposite = new ScrolledComposite(grpName, SWT.H_SCROLL | SWT.V_SCROLL);
        scrolledComposite.setMinWidth(800);
        scrolledComposite.setMinHeight(600);
        scrolledComposite.setExpandHorizontal(true);
        scrolledComposite.setExpandVertical(true);

        lblError = new Label(shell, SWT.NONE);
        lblError.setForeground(SWTResourceManager.getColor(SWT.COLOR_RED));
        lblError.setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, true, false, 1, 1));
        lblError.setVisible(false);

        btnNext = new Button(shell, SWT.NONE);
        btnNext.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, false, false, 1, 1));
        btnNext.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(final SelectionEvent e) {
                next();
            }
        });
        btnNext.setText("&Continue");

    }

    public void setError(final String text) {
        this.error = text;
    }

    /**
     * Triggered by {@link btnNext}. Checks if all forms are valid and retrieves the answers or
     * displays the error message if something is invalid.
     */
    void next() {
        lblError.setVisible(false);
        if (form.validate()) {
            result = form.getAnswers();
            shell.dispose();
        } else {
            lblError.setText(error);
            lblError.setVisible(true);
            shell.layout();
        }
    }
}
// CHECKSTYLE:ON
