package org.eclipse.expi.forms;

import java.util.ArrayList;
import java.util.Collection;

import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Text;
import org.eclipse.wb.swt.SWTResourceManager;

/**
 * Is used to ask "general" questions before the start of the experiment. e.g.: number, email,
 * eclipse experience, java skill
 *
 * @author ojlr
 *
 */
public class FormGeneral extends Composite implements Form {
    private static final int FONT_SIZE = 9;
    private static final int NUM_COLUMNS = 3;

    /** Used to enter the email address. */
    private final Text       txtMail;
    /** Used to enter "the number". */
    private final Text       sNumber;
    /** Used to enter the java skill. */
    private final FormSlider sliderSkill;
    /** Used to enter the whether the user has experience in eclipse or not. */
    private final FormBool   boolExp;
    /** A reference to die containing dialog for error reporting. */
    private final FormDialog fd;

    /**
     * Create the composite.
     *
     * @param parent
     * @param style
     */
    public FormGeneral(final FormDialog fd, final int style) {
        super(fd.getFormComposite(), style);
        this.fd = fd;
        setLayout(new GridLayout(NUM_COLUMNS, false));
        new Label(this, SWT.NONE);

        final Label lblNumber = new Label(this, SWT.NONE);
        lblNumber.setText("Number:");

        sNumber = new Text(this, SWT.BORDER);
        sNumber.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, false, false, 1, 1));
        final Color defaultColor = sNumber.getForeground();
        sNumber.addListener(SWT.CHANGED, new Listener() {
            @Override
            public void handleEvent(final Event event) {
                sNumber.setForeground(defaultColor);
            }
        });

        final Label lbloptional = new Label(this, SWT.NONE);
        lbloptional.setFont(SWTResourceManager.getFont(
                SWTResourceManager.getDefaultFontName(), FONT_SIZE, SWT.ITALIC));
        lbloptional.setText("(optional)");

        final Label lblEmail = new Label(this, SWT.NONE);
        lblEmail.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, false, false, 1, 1));
        lblEmail.setText("E-mail:");

        txtMail = new Text(this, SWT.BORDER);
        txtMail.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, false, false, 1, 1));

        sliderSkill = new FormSlider(this, SWT.NONE, "Java Skills",
                "How experienced are you with java programming?");
        sliderSkill.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, false, false,
                NUM_COLUMNS, 1));

        boolExp = new FormBool(this, SWT.NONE, "Do you have eclipse experience?");
        boolExp.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false, NUM_COLUMNS, 1));
        new Label(this, SWT.NONE);
        new Label(this, SWT.NONE);
        new Label(this, SWT.NONE);
    }

    @Override
    protected void checkSubclass() {
        // Disable the check that prevents subclassing of SWT components
    }

    /**
     * This implementation checks if the number has the right format (a or b and some digits). The
     * email address is an optional answer and the other forms have default values.
     */
    @Override
    public boolean validate() {
        final String txt = sNumber.getText();
        if (txt == null || !txt.matches("[abAB][0-9]+")) {
            fd.setError("Number has to match `[abAB][0-9]+`.");
            sNumber.setForeground(SWTResourceManager.getColor(SWT.COLOR_RED));
            return false;
        }
        if (!sliderSkill.validate()) {
            return false;
        }
        if (!boolExp.validate()) {
            fd.setError("Please specify if you have eclipse experience.");
            return false;
        }
        return true;
    }

    @Override
    public Collection<Answer> getAnswers() {
        final ArrayList<Answer> r = new ArrayList<Answer>(4);
        r.add(new Answer("number", sNumber.getText()));
        if (txtMail.getText().length() > 0) {
            r.add(new Answer("mail", txtMail.getText()));
        }
        r.addAll(sliderSkill.getAnswers());
        r.addAll(boolExp.getAnswers());

        org.eclipse.expi.Activator.getDefault().setNumber(sNumber.getText());

        return r;
    }

}
