package org.eclipse.expi.forms;

import java.util.Map;

import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;

/**
 * This form is used to rate the workload of a task.
 *
 * @author ojlr
 *
 */
public class FormRating extends Composite {

    private static final int NUM_COLUMNS = 5;
    private final Label  lblLeft;
    private final Label  lblRight;
    private final Button btnLeft;
    private final Button btnRight;
    private final Button btnNext;
    private final String tipLeft;
    private final String tipRight;

    /**
     * Create the composite.
     *
     * @param parent
     *            The containing composite.
     * @param style
     *            The swt style.
     * @param descriptions
     *            The names and the descriptions of the different categories.
     * @param left
     *            The name of the left category.
     * @param right
     *            The name of the right category.
     * @param nextListener
     *            The listener, which is triggered if the next/continue button is activated.
     */
    public FormRating(final Composite parent, final int style,
            final Map<String, String> descriptions, final String left, final String right,
            final Listener nextListener) {
        super(parent, SWT.NONE);
        setLayout(new GridLayout(NUM_COLUMNS, false));

        tipLeft = descriptions.get(left);
        tipRight = descriptions.get(right);

        lblLeft = new Label(this, SWT.NONE);
        lblLeft.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, true, false, 1, 1));
        lblLeft.setText(left);
        lblLeft.setToolTipText(tipLeft);

        btnLeft = new Button(this, SWT.RADIO | SWT.CENTER);
        btnLeft.setAlignment(SWT.CENTER);
        btnLeft.setLayoutData(new GridData(SWT.CENTER, SWT.CENTER, false, false, 1, 1));
        btnLeft.setToolTipText(tipLeft);

        btnRight = new Button(this, SWT.RADIO | SWT.CENTER);
        btnRight.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, false, false, 1, 1));
        btnRight.setToolTipText(tipRight);

        lblRight = new Label(this, SWT.NONE);
        lblRight.setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, true, false, 1, 1));
        lblRight.setText(right);
        lblRight.setToolTipText(tipRight);

        btnNext = new Button(this, SWT.NONE);
        btnNext.addListener(SWT.Selection, nextListener);
        btnNext.setText("&Continue");
        btnNext.setEnabled(false);

        final Listener enableListener = new Listener() {
            @Override
            public void handleEvent(final Event event) {
                btnNext.setEnabled(true);
            }
        };
        btnLeft.addListener(SWT.Selection, enableListener);
        btnRight.addListener(SWT.Selection, enableListener);
    }

    public String getTipLeft() {
        return tipLeft;
    }

    public String getTipRight() {
        return tipRight;
    }

    public String getRight() {
        return lblRight.getText();
    }

    public String getLeft() {
        return lblLeft.getText();
    }

    @Override
    protected void checkSubclass() {
        // Disable the check that prevents subclassing of SWT components
    }

    @Override
    public void setEnabled(final boolean b) {
        super.setEnabled(b);
        if (!b) {
            btnNext.setEnabled(b);
            lblLeft.setEnabled(b);
            lblRight.setEnabled(b);
            btnLeft.setEnabled(b);
            btnRight.setEnabled(b);
            lblLeft.setVisible(b);
            lblRight.setVisible(b);
            btnLeft.setVisible(b);
            btnRight.setVisible(b);
        }
    }

    /**
     *
     * @return The name of the selected category.
     */
    public String getVote() {
        if (btnLeft.getSelection()) {
            return lblLeft.getText();
        } else if (btnRight.getSelection()) {
            return lblRight.getText();
        } else {
            return null;
        }
    }

}
