package org.eclipse.expi.forms;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;

import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.wb.swt.SWTResourceManager;

/**
 * This form is used to compare multiple categories after each other.
 * 
 * @author ojlr
 * 
 */
public class FormRatingTotal extends Composite implements Form {
    private static final int FONT_SIZE = 11;
    private final Map<String, Integer>     count   = new HashMap<String, Integer>();
    private FormRating                     form;
    private final FormDialog               fd;
    private final List<FormRating>         ratings = new LinkedList<FormRating>();
    private final ListIterator<FormRating> ratingIterator;
    private final Label                    leftHelp;
    private final Label                    rightHelp;
    private final Label                    leftHelpDescription;
    private final Label                    rightHelpDescription;
    private final Group                    helpGroup;

    /**
     * Create the composite.
     *
     * @param parent
     * @param style
     */
    public FormRatingTotal(final FormDialog fd, final int style,
            final Map<String, String> descriptions) {
        super(fd.getFormComposite(), style);
        this.fd = fd;
        fd.setNextEnabled(false);
        final RowLayout layout = new RowLayout(SWT.VERTICAL);
        layout.fill = true;
        layout.wrap = false;
        setLayout(layout);
        final Label lblInstruction = new Label(this, SWT.WRAP);

        lblInstruction.setText(""
                + "Now, after you have finished all tasks, we need to know which\n"
                + "source of workload had the biggest impact on the workload you\n"
                + "experienced.\n"
                + "For the following pairs of sources of workload choose the member\n"
                + "that had the bigger impact of both.");

        lblInstruction.setFont(SWTResourceManager.getFont(
                SWTResourceManager.getDefaultFontName(), FONT_SIZE, SWT.ITALIC));
        final String[][] pairs = new String[][] { { "Mental Demand", "Effort" },
                { "Mental Demand", "Frustration" }, { "Effort", "Frustration" } };

        for (final String[] pair : pairs) {
            count.put(pair[0], 0);
            count.put(pair[1], 0);

            final FormRating formRating = new FormRating(this, SWT.NONE, descriptions,
                    pair[0], pair[1], new Listener() {
                        @Override
                        public void handleEvent(final Event event) {
                            next();
                        }
                    });
            formRating.setVisible(false);
            ratings.add(formRating);
        }

        helpGroup = new Group(this, 0);
        helpGroup.setText("Help");
        final GridLayout helpGroupLayout = new GridLayout();
        helpGroupLayout.numColumns = 2;
        helpGroup.setLayout(helpGroupLayout);
        leftHelp = new Label(helpGroup, 0);
        leftHelpDescription = new Label(helpGroup, 0);
        rightHelp = new Label(helpGroup, 0);
        rightHelpDescription = new Label(helpGroup, 0);

        fd.getFormComposite().setMinSize(getSize());
        layout();
        ratingIterator = ratings.listIterator();

        next();
    }

    private void next() {
        if (form != null) {
            final String vote = form.getVote();
            final Integer value = count.get(vote);
            count.put(vote, value + 1);
            form.setEnabled(false);
        }
        if (ratingIterator.hasNext()) {
            form = ratingIterator.next();
            setHelp(form);
            form.setVisible(true);
            form.setEnabled(true);
            form.setFocus();
        } else {
            fd.setNextEnabled(true);
        }
    }

    void setHelp(final FormRating f) {
        leftHelp.setText(f.getLeft());
        leftHelpDescription.setText(f.getTipLeft());
        rightHelp.setText(f.getRight());
        rightHelpDescription.setText(f.getTipRight());
        helpGroup.layout();
        helpGroup.pack();
    }

    @Override
    protected void checkSubclass() {
        // Disable the check that prevents subclassing of SWT components
    }

    @Override
    public boolean validate() {
        return true;
    }

    @Override
    public Collection<Answer> getAnswers() {
        final ArrayList<Answer> r = new ArrayList<Answer>(count.size());
        for (final Map.Entry<String, Integer> e : count.entrySet()) {
            r.add(new Answer(e.getKey(), e.getValue().toString()));
        }
        return r;
    }
}
