package org.eclipse.expi.views;

import java.io.FileWriter;
import java.io.IOException;
import java.net.URLEncoder;
import java.text.NumberFormat;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Stack;
import java.util.StringTokenizer;
import java.util.Timer;
import java.util.TimerTask;

import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.expi.AbstractProgramLauncher;
import org.eclipse.expi.FileLogger;
import org.eclipse.expi.ProgramLauncherJDT;
import org.eclipse.expi.TaskProcessor;
import org.eclipse.expi.forms.Answer;
import org.eclipse.expi.forms.FormDialog;
import org.eclipse.expi.forms.FormFactory;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.part.ViewPart;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

/**
 */

public class ExpView extends ViewPart {

    private static final long                   MILLISECONDS_PER_SECOND = 1000L;
    private static final int                    SECONDS_PER_MINUTE      = 60;
    private static final int                    MINUTES_PER_HOUR        = 60;
    private static final int                    TIMER_PERIOD            = 1000;
    private static final int                    TIMER_DELAY             = 1000;
    private static final int                    HEIGHT                  = 200;
    private static final int                    WIDTH                   = 200;
    private static final int                    MODE_MIN                = 1;
    private static final int                    MODE_MAX                = 2;

    private NumberFormat                        nf;
    private Composite                           parent;
    private TaskPlayer                          player;
    private Label                               questionNoLabel;
    private long                                startTime;
    private StyledText                          answerText;
    private AbstractProgramLauncher             launcher;
    private TaskProcessor                       tc;
    private final Timer                         timer;
    private final Map<MyTimerTask, MyTimerTask> mTimers;
    private final Stack<String>                 timerIdStack;
    private String                              formName;
    private boolean                             formModal;

    public ExpView() {
        timer = new Timer();
        mTimers = new HashMap<MyTimerTask, MyTimerTask>();
        timerIdStack = new Stack<String>();
    }

    @Override
    public void createPartControl(final Composite p) {
        tc = TaskProcessor.getInstance();
        parent = p;
        nf = NumberFormat.getIntegerInstance();
        nf.setMinimumIntegerDigits(2);
        nf.setMaximumFractionDigits(0);

        createViewContents(p);
        makeActions();
        nextTask();
        p.setSize(WIDTH, HEIGHT);
        final MyTimerTask t = new MyTimerTask("Tick", 0);
        timer.schedule(t, TIMER_DELAY, TIMER_PERIOD);
        startTime = System.currentTimeMillis();
    }

    private void createViewContents(final Composite p) {
        final FormToolkit tk = new FormToolkit(p.getDisplay());

        player = new TaskPlayer(p, tk, 0);
        player.getNextControl().addSelectionListener(new SelectionListener() {
            @Override
            public void widgetSelected(final SelectionEvent e) {
                if (showDial()) {
                    nextTask();
                }
            }

            @Override
            public void widgetDefaultSelected(final SelectionEvent e) {
            }
        });
        launcher = new ProgramLauncherJDT(player.getRunControl());
        player.getRunControl().addSelectionListener(launcher);
    }

    private void makeActions() {
    }

    private boolean showDial() {
        final MessageBox confirmDialog = new MessageBox(Display.getCurrent().getActiveShell(),
                SWT.ICON_QUESTION | SWT.YES | SWT.NO);
        confirmDialog.setMessage("Do you really want to proceed?\n"
                + "You can not come back to this step, once you proceed.");
        confirmDialog.setText("Proceed?");
        return confirmDialog.open() == SWT.YES;
    }

    @Override
    public void setFocus() {
    }

    @Override
    public void dispose() {
        super.dispose();
        timer.cancel();
    }

    private void saveAnswer() {
        if (answerText != null && answerText.isVisible()) {
            log("answer: '" + answerText.getText() + "'");
        }
    }

    private void nextTask() {
        saveAnswer();
        boolean bAgain = true;
        while (bAgain) {
            if (!tc.more()) {
                return;
            }
            tc.next();
            final Node node = tc.getNode();
            if (node.getNodeName().equals("timed")) {
                if (tc.isOpening()) {
                    // create new timer
                    final String id = ((Element) node).getAttribute("id");
                    final String mintime = ((Element) node).getAttribute("mintime");
                    final String maxtime = ((Element) node).getAttribute("maxtime");
                    log("new timer: id=" + id + ", mintime=" + mintime + ", maxtime="
                            + maxtime);
                    timerIdStack.push(id);
                    if (mintime != null && mintime.length() > 0) {
                        addTimer(id, MODE_MIN, mintime);
                    }
                    if (maxtime != null && maxtime.length() > 0) {
                        addTimer(id, MODE_MAX, maxtime);
                    }

                    final String showtimerString = ((Element) node).getAttribute("showtimer");
                    final boolean showtimer = Boolean.valueOf(showtimerString);
                    player.setTimerVisible(showtimer);
                } else {
                    // close maxtime timer
                    final String id = timerIdStack.pop();
                    MyTimerTask mtt = mTimers.remove(new MyTimerTask(id, MODE_MAX));
                    if (mtt != null) {
                        mtt.cancel();
                        mTimers.remove(mtt);
                    }
                    // wait for mintime timer
                    mtt = mTimers.get(new MyTimerTask(id, MODE_MIN));
                    if (mtt != null) {
                        log("waiting for mintime expiration, id=" + id);

                        player.setContent("Waiting for expiration of minimum assigned time...");
                        player.getNextControl().setVisible(false);
                        bAgain = false;
                    }
                }
            } else if (node.getNodeName().equals("info")) {
                if (tc.isOpening()) {
                    player.setContent(node.getTextContent());
                    player.getNextControl().setVisible(!tc.getProgressInfo().isFinished());
                    log("info: '" + node.getTextContent() + "'");
                    bAgain = false;

                    // Allow to launch the program associated with this element.
                    launcher.setProgram(((Element) node).getAttribute("program"));
                    String name = ((Element) node).getAttribute("name");
                    if (name != null) {
                        name = name.trim();
                        if (name.isEmpty()) {
                            name = null;
                        }
                    }
                    player.setRunControlText(name);
                }
            } else if (node.getNodeName().equals("question")) {
                if (tc.isOpening()) {
                    // Removed handling questions here.
                    bAgain = false;
                }
            } else if (node.getNodeName().equals("resetTimer")) {
                startTime = System.currentTimeMillis();
            } else if (node.getNodeName().equals("form")) {
                player.setContent("");
                if (tc.isOpening()) {
                    formName = node.getTextContent();
                    final String modal = ((Element) node).getAttribute("modal");
                    formModal = Boolean.valueOf(modal);
                } else {
                    bAgain = false;
                    if (!formModal) {
                        player.setContent("You have to complete the form before you can proceed.");
                        player.getNextControl().setEnabled(false);
                    }
                    final FormDialog fd = FormFactory.createDialog(parent, formName,
                            formModal);
                    log("Open " + formName);
                    writeResults(formName, fd.open());
                    log("Closed " + formName);
                    player.getNextControl().setEnabled(true);
                    nextTask();
                }
            }
        }
    }

    /**
     * Writes the results of the forms of a dialog to a file named accordingly.
     *
     * @param form
     *            The name of the form.
     * @param answers
     *            The answers.
     */
    private void writeResults(final String form, final Collection<Answer> answers) {
        try {
            final FileWriter fw = new FileWriter(ResourcesPlugin.getWorkspace().getRoot()
                    .getLocation().toString()
                    + "/" + form.replace(' ', '_') + ".xml", true);
            fw.write("<form>\n");
            final String enc = System.getProperty("file.encoding");
            for (final Answer a : answers) {
                fw.write(String.format("<answer name=\"%s\">\n%s\n</answer>\n",
                        URLEncoder.encode(a.getName(), enc),
                        URLEncoder.encode(a.getValue(), enc)));
            }
            fw.write("</form>\n");
            fw.flush();
            fw.close();
        } catch (final IOException e) {
            FileLogger.getLogger("error").log(e);
        }
    }

    private void skipToTimer(final String termId) {
        saveAnswer();
        log("skipping to timer " + termId);
        boolean bAgain = true;
        while (bAgain) {
            if (!tc.more()) {
                return;
            }
            tc.next();
            final Node node = tc.getNode();
            if (node.getNodeName().equals("timed")) {
                if (tc.isOpening()) {
                    // create new timer
                    final String id = ((Element) node).getAttribute("id");
                    timerIdStack.push(id);
                } else {
                    // close maxtime timer
                    final String id = timerIdStack.pop();
                    if (id.equals(termId)) {
                        bAgain = false;
                    }

                    MyTimerTask mtt = mTimers.remove(new MyTimerTask(id, MODE_MAX));
                    if (mtt != null) {
                        mtt.cancel();
                        mTimers.remove(mtt);
                    }

                    mtt = mTimers.remove(new MyTimerTask(id, MODE_MIN));
                    if (mtt != null) {
                        mtt.cancel();
                        mTimers.remove(mtt);
                    }
                }
            } else if (node.getNodeName().equals("question")) {
                updateQuestionNoLabel();
            } else if (node.getNodeName().equals("resetTimer")) {
                startTime = System.currentTimeMillis();
            }
        }
    }

    private void updateQuestionNoLabel() {
        final TaskProcessor.ProgressInfo pi = tc.getProgressInfo();
        questionNoLabel.setText("Question " + pi.getNumCurrentQuestion() + " of "
                + pi.getNumNumQuestions());
    }

    private void addTimer(final String id, final int mode, final String delay) {
        final StringTokenizer st = new StringTokenizer(delay, ":");
        final int hours = Integer.parseInt(st.nextToken());
        final int min = Integer.parseInt(st.nextToken());
        final int sec = Integer.parseInt(st.nextToken());
        final long d = (((hours * MINUTES_PER_HOUR) + min) * SECONDS_PER_MINUTE + sec)
                * MILLISECONDS_PER_SECOND;
        final MyTimerTask t = new MyTimerTask(id, mode);
        mTimers.put(t, t);
        timer.schedule(t, d);
    }

    private void timerNotification(final MyTimerTask t) {
        parent.getDisplay().syncExec(new TimedTask(t));
    }

    /**
     * Writes to interaction.log
     *
     * @param msg
     *            The message that should be logged.
     */
    private static void log(final String msg) {
        FileLogger.getLogger("interaction").log(msg);
    }

    /**
     * Is used to register a timer that updates the (invisible) remaining time.
     */
    private final class TimedTask implements Runnable {
        private final MyTimerTask t;

        private TimedTask(final MyTimerTask t) {
            this.t = t;
        }

        @Override
        public void run() {
            try {
                if (t.getId().equals("Tick")) {
                    final long time = (System.currentTimeMillis() - startTime)
                            / MILLISECONDS_PER_SECOND;
                    String timeStr = nf.format(time / SECONDS_PER_MINUTE) + ":"
                            + nf.format(time % SECONDS_PER_MINUTE);
                    if (time / SECONDS_PER_MINUTE > SECONDS_PER_MINUTE) {
                        timeStr = (time / (SECONDS_PER_MINUTE * MINUTES_PER_HOUR))
                                + ":"
                                + nf.format((time / SECONDS_PER_MINUTE)
                                        % SECONDS_PER_MINUTE) + ":"
                                + nf.format(time % SECONDS_PER_MINUTE);
                    }
                    player.setTimerText(timeStr);
                } else {
                    // timer expired
                    log("timerEvent: " + t);
                    mTimers.remove(t);
                    if (t.getMode() == MODE_MIN) {
                        if (!player.getNextControl().isVisible()) {
                            log("min. time expired");
                            player.getNextControl().setVisible(true);
                            nextTask();
                        }
                    } else if (t.getMode() == MODE_MAX) {
                        log("time expired");
                        MessageDialog.openInformation(parent.getShell(), "Time expired",
                                "The maximum allowable time for solving"
                                        + "this task has expired.\n"
                                        + "Skipping to the next task.");
                        skipToTimer(t.getId());
                        nextTask();
                    }
                }
            } catch (final Exception e) {
                FileLogger.getLogger("error").log(e);
            }
        }
    }

    private class MyTimerTask extends TimerTask {

        private final String id;
        private final int    mode;

        public MyTimerTask(final String id, final int mode) {
            this.id = id;
            this.mode = mode;
        }

        public String getId() {
            return id;
        }

        public int getMode() {
            return mode;
        }

        @Override
        public void run() {
            timerNotification(this);
        }

        @Override
        public int hashCode() {
            return id.hashCode() ^ mode;
        }

        @Override
        public boolean equals(final Object obj) {
            final MyTimerTask mtt = (MyTimerTask) obj;
            return mtt.id.equals(id) && mtt.mode == mode;
        }

        @Override
        public String toString() {
            return "id=" + id + ", mode=" + mode;
        }
    }
}