package pacman.actors;

import java.awt.Color;
import java.awt.Graphics2D;

import pacman.ai.AIManager;
import pacman.game.GameObject;
import pacman.map.Map;
import pacman.state.StateGame;
import pacman.util.Direction;
import pacman.util.RequestedDirectionBuffer;

/**
 * An actor is any object that has a degree of autonomy or intelligent input
 * (Human / AIManager) dictating the object's behavior ingame Subclass of
 * GameObject
 * 
 * @author Ramsey Kant
 */
public abstract class Actor extends GameObject {

    /** Whether the actor is alive or not. */
    protected boolean isDead;

    /** The x-position on the map where the actor gets spawned */
    protected int spawnX;
    
    /** The y-position on the map where the actor gets spawned */
    protected int spawnY;

    /** The direction in which the actor is currently oriented. */
    protected Direction currentMoveDir;
    
    /** The actor's direction requested by user input. Uses a buffer to
     *  remember requests over some steps to simplify timing for the user.
     *  Computer-controlled actors (ghosts) ignore this. */
    protected RequestedDirectionBuffer requestedMoveDirBuffer;
    
    /** The size of the direction request buffer. */
    private final static int DIRECTION_QUEUE_SIZE = 6;

    /** The current orientation angle of the actor. Ignored by actors that do
     *  not have to orient (aka ghosts). */
    protected int dirOrient;
    
    /** The x-position delta to the current map cell, caused by movement
     *  in pixels. */
    protected float deltaX;
    
    /** The y-position delta to the current map cell, caused by movement
     *  in pixels. */
    protected float deltaY;
    
    /** The movement speed of the actor */
    protected float speed;

    /**
     * Actor Class Constructor
     * 
     * @param type
     *            Object type that is an actor
     * @param color
     *            Base color of the actor
     * @param m
     *            Reference to the global map object
     * @param x
     *            X coordinate to spawn the actor at
     * @param y
     *            Y coordinate to spawn the actor at
     * @see GameObject
     */
    public Actor(int type, Color color, Map m, int x, int y) {
        super(type, color, m, x, y);

        isDead = false;

        // Movement
        spawnX = x;
        spawnY = y;
        currentMoveDir = Direction.none;
        requestedMoveDirBuffer = new RequestedDirectionBuffer(DIRECTION_QUEUE_SIZE);
        dirOrient = 0;
        deltaX = 0;
        deltaY = 0;
        speed = (float) (5d * map.SCALE);
    }

    // Getters and Setters

    /**
     * Returns the original X coordinate the Actor was given in the constructor
     * 
     * @return the X coordinate of the spawn point
     */
    public int getSpawnX() {
        return spawnX;
    }

    /**
     * Returns the original Y coordinate the Actor was given in the constructor
     * 
     * @return the Y coordinate of the spawn point
     */
    public int getSpawnY() {
        return spawnY;
    }

    /**
     * Set the death status of the actor. Used by StateGame and AIManager to
     * determine if the player / ghost has died
     */
    public void setDead(boolean s) {
        isDead = s;
    }

    /**
     * Get dead status
     * 
     * @return True if dead, false if alive
     * @see Actor#setDead(boolean)
     */
    public boolean isDead() {
        return isDead;
    }

    /**
     * Speed is the number of pixels an actor moves across the screen in a given
     * cycle. A full position change is the number of pixels defined in
     * Map.CELL_SIZE
     * 
     * @param s
     *            New Speed
     */
    public void setSpeed(float s) {
        speed = s;
    }

    /**
     * Get the current speed of the actor
     * 
     * @return Current speed
     * @see Actor#setSpeed(float)
     */
    public float getSpeed() {
        return speed;
    }

    /**
     * Set the direction actor should travel in. Player uses this to determine
     * the direction to "auto-move" to Ghosts ignore what is set by this
     * function because their direction is determined within act() based on the
     * path
     */
    public void setMoveDirection(Direction dir) {
        requestedMoveDirBuffer.setRequestedDirection(dir);
    }

    // Public Methods

    /**
     * Attempt to move the actor to the given x,y location. This method will
     * check if a coordinate is valid with the Map class method canMove(). It is
     * not necessary to call canMove() before this function
     * 
     * @param x
     *            A x coordinate to move to
     * @param y
     *            A y coordinate to move to
     * @return True if the move succeeded. False if otherwise
     * @see Map#canMove(Actor, int, int)
     */
    public boolean move(int x, int y) {
        final boolean res = map.canMove(this, x, y);
        if (res) {
            positionX = x;
            positionY = y;
        }
        return res;
    }

    /**
     * The primary logic function for actors. StateGame calls this for players
     * directly in logic() and the AIManager calls this for ghosts in process()
     * 
     * @see GameObject#act()
     * @see StateGame#logic()
     * @see AIManager#process()
     */
    @Override
    public abstract void act();

    /**
     * 
     * @see GameObject#paint(java.awt.Graphics2D)
     */
    @Override
    public abstract void paint(Graphics2D g);

}
