package pacman.util;
import java.util.LinkedList;

/**
 * This buffer is used to remember direction changes requested by the user for
 * some game turns. If a direction change was requested too early (move not
 * possible yet, because the crossing has not been reached) the direction change
 * can be retired in the next steps. The size parameter of the constructor
 * contols how long the user input will be remembered. If another direction is
 * set, all previous directions will be forgotten.
 * 
 * By default the buffer will return the neutral Direction.none direction. If
 * another direction is set, the buffer will return this direction for the next
 * n calls to getRequestedDirection(), where n = size. When a new direction is
 * set, all previously set directions are forgotten.
 * 
 */
public class RequestedDirectionBuffer {

    private final int size;
    private final LinkedList<Direction> queue;

    public RequestedDirectionBuffer(int size) {
        super();
        this.size = size;
        queue = new LinkedList<Direction>();
        fill(Direction.none);
    }

    /**
     * Get the currently requested direction.
     * 
     * @return the currently requested direction.
     */
    public Direction getRequestedDirection() {
        final Direction dir = queue.poll();
        queue.add(Direction.none);
        return dir;
    }

    /**
     * Set the requested direction
     * 
     * @param dir
     *            the requested direction
     */
    public void setRequestedDirection(Direction dir) {
        fill(dir);
    }

    /**
     * Fill the queue with a direction
     * 
     * @param dir
     *            the direction.
     */
    private void fill(Direction dir) {
        queue.clear();
        for (int i = 0; i < size; i++) {
            queue.add(dir);
        }
    }
}
