package pacman.actors;

import java.awt.Color;
import java.awt.Graphics2D;

import pacman.ai.AIManager;
import pacman.game.GameObject;
import pacman.map.Map;
import pacman.map.Path;
import pacman.util.Direction;

/**
 * The Ghost class is the primary enemy in Pacman. Intelligent decisions of
 * ghosts are made by the AIManager class Ghost is a subclass of Actor
 * 
 * @author Ramsey Kant
 */
public class Ghost extends Actor {
    // Movement
    private Path path;
    private int nextStepIdx;
    private boolean needNewPath;

    // State
    private boolean trapped;
    private boolean inFear;
    private boolean debugDrawPath;

    /**
     * Class constructor
     * 
     * @param color
     *            Color of the ghost's 'body'
     * @param m
     *            Reference to the map
     * @param x
     *            X coordinate to spawn at
     * @param y
     *            Y coordinate to spawn at
     * @param trap
     *            Set trapped status
     */
    public Ghost(Color color, Map m, int x, int y, boolean trap) {
        super(GameObject.OBJECT_GHOST, color, m, x, y);
        needNewPath = true;
        inFear = false;
        trapped = trap;
        debugDrawPath = false;
    }

    /**
     * Return the fear status of the ghost
     * 
     * @return True if fearful
     */
    public boolean isInFear() {
        return inFear;
    }

    /**
     * Set fear status. AIManager interperates this setting for behavior
     * 
     * @param f
     *            Fear status, true if fearful
     */
    public void setFear(boolean f) {
        inFear = f;
    }

    /**
     * Get the current trapped status
     * 
     * @return True if the ghost is currently in the spawn-jail
     */
    public boolean isTrapped() {
        return trapped;
    }

    /**
     * Set the current trapped status
     * 
     * @param t
     *            Trye uf the ghost is in the spawn-jail
     */
    public void setTrapped(boolean t) {
        trapped = t;
    }

    /**
     * Flag that is set to true when the path reaches the last possible step
     * 
     * @return True if the AIManager needs to assign a new path
     */
    public boolean needsNewPath() {
        return needNewPath;
    }

    /**
     * Update the Path object for the ghost to follow'
     * 
     * @param p
     *            Path object generated in process() by the AIManager
     * @see AIManager#process()
     */
    public void updatePath(Path p) {
        nextStepIdx = 1;
        path = p;
        needNewPath = false;
    }

    /**
     * Direct's the paint() function to draw the current path of the ghost on
     * the map
     * 
     * @param d
     *            If true, debug is on and the path will be drawn
     * @see AIManager#setDebugEnabled
     */
    public void setDebugDrawPath(boolean d) {
        debugDrawPath = d;
    }

    /**
     * Run a think cycle for the AI. Major decisions are made by the AIManager
     * (pathing), this just determines movement and screen draw deltas
     * 
     * @see Actor#act()
     */
    @Override
    public void act() {
        // Move to the next step
        if (path != null && nextStepIdx < path.getLength()) {

            // Figure out the direction
            if ((path.getY(nextStepIdx) - positionY) < 0) {
                currentMoveDir = Direction.up;
            } else if ((path.getY(nextStepIdx) - positionY) > 0) {
                currentMoveDir = Direction.down;
            } else if ((path.getX(nextStepIdx) - positionX) > 0) {
                currentMoveDir = Direction.right;
            } else {
                currentMoveDir = Direction.left;
            }

            // Based on the direction, move the screen delta's and the X,Y
            // coordinates if the # of pixels for the cell have been surpassed
            switch (currentMoveDir) {
                case up:
                    deltaX = 0;
                    deltaY = deltaY - speed;
                    // If the movement delta has surpassed the number of pixels for
                    // the cell, set him to the map cell he has reached by his movement.
                    if (Math.abs(deltaY) >= map.CELL_SIZE) {
                        deltaY = 0;
                        move(positionX, positionY + 1);
                        nextStepIdx++;
                    }
                    break;
                case right:
                    deltaX = deltaX + speed;
                    deltaY = 0;
                    if (Math.abs(deltaX) >= map.CELL_SIZE) {
                        deltaX = 0;
                        move(positionX + 1, positionY);
                        nextStepIdx++;
                    }
                    break;
                case down:
                    deltaX = 0;
                    deltaY = deltaY + speed;
                    if (Math.abs(deltaY) >= map.CELL_SIZE) {
                        deltaY = 0;
                        move(positionX, positionY + 1);
                        nextStepIdx++;
                    }
                    break;
                case left:
                    deltaX = deltaX - speed;
                    deltaY = 0;
                    if (Math.abs(deltaX) >= map.CELL_SIZE) {
                        deltaX = 0;
                        move(positionX + 1, positionY);
                        nextStepIdx++;
                    }
                    break;
                case stop:
                case none:
                    // do not move
            }
        } else {
            needNewPath = true;
        }
    }

    /**
     * Draw the ghost
     * 
     * @see GameObject#paint(Graphics2D)
     */
    @Override
    public void paint(Graphics2D g) {
        // Change the position of pacman on screen by the offsets m_fDelta
        final int screenX = (int) ((map.CELL_SIZE * positionX) + deltaX);
        final int screenY = (int) ((map.CELL_SIZE * positionY) + deltaY);

        g.setColor(objColor);

        // Body
        if (inFear) {
            g.setColor(Color.WHITE);
        }
        g.fillArc(screenX, screenY, map.CELL_SIZE, map.CELL_SIZE, 0, 360);
        g.fillRect((int) ((map.CELL_SIZE * positionX) + deltaX), (int) ((map.CELL_SIZE * positionY)
                + (map.CELL_SIZE / 2) + deltaY), map.CELL_SIZE, map.CELL_SIZE / 2);

        // Eyes
        if (inFear) {
            g.setColor(Color.BLACK);
        } else {
            g.setColor(Color.WHITE);
        }
        g.fillOval((int) ((map.CELL_SIZE * positionX) + 4 + deltaX),
                (int) ((map.CELL_SIZE * positionY) + 3 + deltaY), 8, 10);
        g.fillOval((int) ((map.CELL_SIZE * positionX) + 12 + deltaX),
                (int) ((map.CELL_SIZE * positionY) + 3 + deltaY), 8, 10);

        // Eyeballs
        g.setColor(Color.BLUE);
        g.fillOval((int) ((map.CELL_SIZE * positionX) + 7 + deltaX),
                (int) ((map.CELL_SIZE * positionY) + 6 + deltaY), 4, 4);
        g.fillOval((int) ((map.CELL_SIZE * positionX) + 13 + deltaX),
                (int) ((map.CELL_SIZE * positionY) + 6 + deltaY), 4, 4);

        // Debug draw path
        if (debugDrawPath && path != null) {
            for (int i = 0; i < path.getLength(); i++) {
                final Path.Step s = path.getStep(i);
                g.setColor(objColor);
                g.drawLine(map.CELL_SIZE * s.getX(), map.CELL_SIZE * s.getY(),
                        (map.CELL_SIZE * s.getX()) + map.CELL_SIZE, (map.CELL_SIZE * s.getY())
                                + map.CELL_SIZE);
            }
        }
    }
}
