import java.awt.Color;
import java.awt.Graphics2D;

/**
 * The Ghost class is the primary enemy in Pacman. Intelligent decisions of ghosts are made by the AIManager class
 * Ghost is a subclass of Actor
 * 
 * @author Ramsey Kant
 */
public class Ghost extends Actor {
	// Movement
	private Path path;
	private int nextStepIdx;
	private boolean needNewPath;
	
	// State
	private boolean trapped;
	private boolean inFear;
	private boolean debugDrawPath;
	
	/**
	 * Class constructor
	 * 
	 * @param color Color of the ghost's 'body'
	 * @param map Reference to the map
	 * @param x X coordinate to spawn at
	 * @param y Y coordinate to spawn at
	 * @param trap Set trapped status
	 */
	public Ghost(Color color, Map map, int x, int y, boolean trap) {
		super(GameObject.OBJECT_GHOST, color, map, x, y);
		needNewPath = true;
		inFear = false;
		trapped = trap;
		debugDrawPath = false;
	}
	
	/**
	 * Return the fear status of the ghost
	 * 
	 * @return True if fearful
	 */
	public boolean isInFear() {
		return inFear;
	}
	
	/**
	 * Set fear status. AIManager interperates this setting for behavior
	 * 
	 * @param f Fear status, true if fearful
	 */
	public void setFear(boolean f) {
		inFear = f;
	}
	
	/**
	 * Get the current trapped status
	 * 
	 * @return True if the ghost is currently in the spawn-jail
	 */
	public boolean isTrapped() {
		return trapped;
	}
	
	/**
	 * Set the current trapped status
	 * 
	 * @param t Trye uf the ghost is in the spawn-jail
	 */
	public void setTrapped(boolean t) {
		trapped = t;
	}
	
	/**
	 * Flag that is set to true when the path reaches the last possible step
	 * 
	 * @return True if the AIManager needs to assign a new path
	 */
	public boolean needsNewPath() {
		return needNewPath;
	}
	
	/**
	 * Update the Path object for the ghost to follow'
	 * 
	 * @param p Path object generated in process() by the AIManager
	 * @see AIManager#process()
	 */
	public void updatePath(Path p) {
		nextStepIdx = 1;
		path = p;
		needNewPath = false;
	}
	
	/**
	 * Direct's the paint() function to draw the current path of the ghost on the map
	 * 
	 * @param d If true, debug is on and the path will be drawn
	 * @see AIManager#setDebugEnabled
	 */
	public void setDebugDrawPath(boolean d) {
		debugDrawPath = d;
	}
	
	/**
	 * Run a think cycle for the AI. Major decisions are made by the AIManager (pathing), this just determines movement and screen draw deltas
	 * 
	 * @see Actor#act()
	 */
	@Override
	public void act() {
		// Move to the next step
		if(path != null && nextStepIdx < path.getLength()) {
			
			// Figure out the direction
			if((path.getY(nextStepIdx)-m_iY) < 0)
				m_iDirection = MOVE_UP;
			else if((path.getY(nextStepIdx)-m_iY) > 0)
				m_iDirection = MOVE_DOWN;
			else if((path.getX(nextStepIdx)-m_iX) > 0)
				m_iDirection = MOVE_RIGHT;
			else
				m_iDirection = MOVE_LEFT;
			
			// Based on the direction, move the screen delta's and the X,Y coordinates if the # of pixels for the cell have been surpassed
			switch(m_iDirection) {
				case MOVE_UP:
					m_fDeltaX = 0;
					m_fDeltaY -= speed;
					// The delta has surpassed the # of pixels for the cell, meaning we can officially change the coordinate
					if(Math.abs(m_fDeltaY) >= g_map.CELL_SIZE) {
						m_fDeltaY = 0;
						move(path.getX(nextStepIdx), path.getY(nextStepIdx));
						nextStepIdx++;
					}
					dirOrient = 90;
					break;
				case MOVE_RIGHT:
					m_fDeltaX += speed;
					m_fDeltaY = 0;
					if(Math.abs(m_fDeltaX) >= g_map.CELL_SIZE) {
						m_fDeltaX = 0;
						move(path.getX(nextStepIdx), path.getY(nextStepIdx));
						nextStepIdx++;
					}
					dirOrient = 0;
					break;
				case MOVE_DOWN:
					m_fDeltaX = 0;
					m_fDeltaY += speed;
					if(Math.abs(m_fDeltaY) >= g_map.CELL_SIZE) {
						m_fDeltaY = 0;
						move(path.getX(nextStepIdx), path.getY(nextStepIdx));
						nextStepIdx++;
					}
					dirOrient = -90;
					break; 
				case MOVE_LEFT:
					m_fDeltaX -= speed;
					m_fDeltaY = 0;
					if(Math.abs(m_fDeltaX) >= g_map.CELL_SIZE) {
						m_fDeltaX = 0;
						move(path.getX(nextStepIdx), path.getY(nextStepIdx));
						nextStepIdx++;
					}
					dirOrient = 180;
					break;
				default:
					// MOVE_NONE (stand still)
					m_fDeltaX = 0;
					m_fDeltaY = 0;
					break;
			}
		} else {
			needNewPath = true;
		}
	}
	
	/**
	 * Draw the ghost
	 * 
	 * @see GameObject#paint(Graphics2D)
	 */
	@Override
	public void paint(Graphics2D g) {
		// Change the position of pacman on screen by the offsets m_fDelta
		int screenX = (int)((g_map.CELL_SIZE * m_iX) + m_fDeltaX);
		int screenY = (int)((g_map.CELL_SIZE * m_iY) + m_fDeltaY);
			
		g.setColor(m_color);
		
		// Body
		if(inFear)
			g.setColor(Color.WHITE);
		g.fillArc(screenX, screenY, g_map.CELL_SIZE, g_map.CELL_SIZE, 0, 360);
		g.fillRect((int)((g_map.CELL_SIZE * m_iX) + m_fDeltaX), (int)((g_map.CELL_SIZE * m_iY)+(g_map.CELL_SIZE/2) + m_fDeltaY), g_map.CELL_SIZE, g_map.CELL_SIZE/2);
		
		// Eyes
		if(inFear)
			g.setColor(Color.BLACK);
		else
			g.setColor(Color.WHITE);
		g.fillOval((int)((g_map.CELL_SIZE * m_iX)+6 + m_fDeltaX), (int)((g_map.CELL_SIZE * m_iY)+4 + m_fDeltaY), 8, 10);
		g.fillOval((int)((g_map.CELL_SIZE * m_iX)+16 + m_fDeltaX), (int)((g_map.CELL_SIZE * m_iY)+4 + m_fDeltaY), 8, 10);
		
		// Eyeballs
		g.setColor(Color.BLUE);
		g.fillOval((int)((g_map.CELL_SIZE * m_iX)+8 + m_fDeltaX), (int)((g_map.CELL_SIZE * m_iY)+6 + m_fDeltaY), 4, 4);
		g.fillOval((int)((g_map.CELL_SIZE * m_iX)+18 + m_fDeltaX), (int)((g_map.CELL_SIZE * m_iY)+6 + m_fDeltaY), 4, 4);
		
		// Debug draw path
		if(debugDrawPath && path != null) {
			for(int i = 0; i < path.getLength(); i++) {
				Path.Step s = path.getStep(i);
				g.setColor(m_color);
				g.drawLine(g_map.CELL_SIZE * s.getX(), g_map.CELL_SIZE * s.getY(), (g_map.CELL_SIZE * s.getX())+g_map.CELL_SIZE, (g_map.CELL_SIZE * s.getY())+g_map.CELL_SIZE);
			}
		}
	}
}
