/*
 * Decompiled with CFR 0.152.
 */
package org.conqat.lib.commons.filesystem;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.EOFException;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.IllegalFormatException;
import java.util.List;
import java.util.Properties;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.JarOutputStream;
import java.util.zip.GZIPInputStream;
import java.util.zip.ZipEntry;
import org.conqat.lib.commons.assertion.CCSMAssert;
import org.conqat.lib.commons.assertion.CCSMPre;
import org.conqat.lib.commons.collections.CollectionUtils;
import org.conqat.lib.commons.filesystem.EByteOrderMark;
import org.conqat.lib.commons.filesystem.FilenameComparator;
import org.conqat.lib.commons.logging.ILogger;
import org.conqat.lib.commons.string.StringUtils;

public class FileSystemUtils {
    public static final String UTF8_ENCODING = "UTF-8";
    public static final Charset UTF8_CHARSET = Charset.forName("UTF-8");

    public static int copy(InputStream input, OutputStream output) throws IOException {
        int len;
        byte[] buffer = new byte[1024];
        int size = 0;
        while ((len = input.read(buffer)) > 0) {
            output.write(buffer, 0, len);
            size += len;
        }
        return size;
    }

    public static void copyFile(File sourceFile, File targetFile) throws IOException {
        FileSystemUtils.ensureParentDirectoryExists(targetFile);
        FileChannel sourceChannel = new FileInputStream(sourceFile).getChannel();
        FileChannel targetChannel = new FileOutputStream(targetFile).getChannel();
        sourceChannel.transferTo(0L, sourceChannel.size(), targetChannel);
        sourceChannel.close();
        targetChannel.close();
    }

    public static void copyFile(String sourceFilename, String targetFilename) throws IOException {
        FileSystemUtils.copyFile(new File(sourceFilename), new File(targetFilename));
    }

    public static int copyFiles(File sourceDirectory, File targetDirectory, FileFilter fileFilter) throws IOException {
        List<File> files = FileSystemUtils.listFilesRecursively(sourceDirectory, fileFilter);
        int fileCount = 0;
        for (File sourceFile : files) {
            if (!sourceFile.isFile()) continue;
            String path = sourceFile.getAbsolutePath();
            int index = sourceDirectory.getAbsolutePath().length();
            String newPath = path.substring(index);
            File targetFile = new File(targetDirectory, newPath);
            FileSystemUtils.copyFile(sourceFile, targetFile);
            ++fileCount;
        }
        return fileCount;
    }

    public static int createJARFile(File jarFile, File directory, FileFilter filter) throws IOException {
        JarOutputStream out = new JarOutputStream(new FileOutputStream(jarFile));
        List<File> files = FileSystemUtils.listFilesRecursively(directory, filter);
        int fileCount = 0;
        for (File file : files) {
            if (!file.isFile()) continue;
            FileInputStream in = new FileInputStream(file);
            String entryName = file.getAbsolutePath().substring(directory.getAbsolutePath().length() + 1);
            entryName = entryName.replace(File.separatorChar, '/');
            out.putNextEntry(new ZipEntry(entryName));
            FileSystemUtils.copy(in, out);
            out.closeEntry();
            in.close();
            ++fileCount;
        }
        out.close();
        return fileCount;
    }

    public static String createRelativePath(File path, File relativeTo) throws IOException {
        CCSMAssert.isNotNull(path, "Path must not be null!");
        CCSMAssert.isNotNull(relativeTo, "relativeTo must not be null!");
        if (!path.isDirectory() || !relativeTo.isDirectory()) {
            throw new IllegalArgumentException("Both arguments must be existing directories!");
        }
        path = path.getCanonicalFile();
        relativeTo = relativeTo.getCanonicalFile();
        HashSet<File> parents = new HashSet<File>();
        File f = path;
        while (f != null) {
            parents.add(f);
            f = f.getParentFile();
        }
        File root = relativeTo;
        while (root != null && !parents.contains(root)) {
            root = root.getParentFile();
        }
        if (root == null) {
            return path.getAbsolutePath();
        }
        String result = "";
        while (!path.equals(root)) {
            result = String.valueOf(path.getName()) + "/" + result;
            path = path.getParentFile();
        }
        while (!relativeTo.equals(root)) {
            result = "../" + result;
            relativeTo = relativeTo.getParentFile();
        }
        return result;
    }

    public static void deleteRecursively(File directory) {
        if (directory == null) {
            throw new IllegalArgumentException("Directory may not be null.");
        }
        if (directory.listFiles() == null) {
            throw new IllegalArgumentException("Argument is not a valid directory.");
        }
        File[] fileArray = directory.listFiles();
        int n = fileArray.length;
        int n2 = 0;
        while (n2 < n) {
            File entry = fileArray[n2];
            if (entry.isDirectory()) {
                FileSystemUtils.deleteRecursively(entry);
            }
            entry.delete();
            ++n2;
        }
        directory.delete();
    }

    public static void deleteFile(File file) throws IOException {
        if (file.exists() && !file.delete()) {
            throw new IOException("Could not delete " + file);
        }
    }

    public static void renameFileTo(File file, File dest) throws IOException {
        if (!file.renameTo(dest)) {
            throw new IOException("Could not rename " + file + " to " + dest);
        }
    }

    public static void mkdir(File dir) throws IOException {
        if (!dir.mkdir()) {
            throw new IOException("Could not create directory " + dir);
        }
    }

    public static void mkdirs(File dir) throws IOException {
        if (!dir.mkdirs()) {
            throw new IOException("Could not create directory " + dir);
        }
    }

    public static void ensureDirectoryExists(File file) throws IOException {
        if (!file.exists() && !file.mkdirs()) {
            throw new IOException("Couldn't create directory: " + file);
        }
    }

    public static void ensureParentDirectoryExists(File file) throws IOException {
        FileSystemUtils.ensureDirectoryExists(file.getCanonicalFile().getParentFile());
    }

    public static List<File> listFilesRecursively(File directory) {
        return FileSystemUtils.listFilesRecursively(directory, null);
    }

    public static List<File> listFilesRecursively(File directory, FileFilter filter) {
        if (directory == null || !directory.isDirectory()) {
            return CollectionUtils.emptyList();
        }
        ArrayList<File> result = new ArrayList<File>();
        FileSystemUtils.listFilesRecursively(directory, result, filter);
        return result;
    }

    public static String getFileExtension(File file) {
        String name = file.getName();
        int posLastDot = name.lastIndexOf(46);
        if (posLastDot < 0) {
            return null;
        }
        return name.substring(posLastDot + 1);
    }

    public static File newFile(File parent, String ... elements) {
        if (elements.length == 0) {
            return parent;
        }
        File child = new File(parent, elements[0]);
        String[] remainingElements = new String[elements.length - 1];
        System.arraycopy(elements, 1, remainingElements, 0, elements.length - 1);
        return FileSystemUtils.newFile(child, remainingElements);
    }

    public static String readFile(File file) throws IOException {
        return FileSystemUtils.readFile(file, Charset.defaultCharset().name());
    }

    public static String readFileUTF8(File file) throws IOException {
        return FileSystemUtils.readFile(file, UTF8_ENCODING);
    }

    public static String readFile(File file, String encoding) throws IOException, UnsupportedEncodingException {
        byte[] buffer = FileSystemUtils.readFileBinary(file);
        EByteOrderMark bom = EByteOrderMark.determineBOM(buffer);
        if (bom != null) {
            return new String(buffer, bom.getBOMLength(), buffer.length - bom.getBOMLength(), bom.getEncoding());
        }
        return new String(buffer, encoding);
    }

    public static List<String> readLines(File file, String encoding) throws UnsupportedEncodingException, IOException {
        return StringUtils.splitLinesAsList(FileSystemUtils.readFile(file, encoding));
    }

    public static List<String> readLinesUTF8(File file) throws UnsupportedEncodingException, IOException {
        return FileSystemUtils.readLines(file, UTF8_ENCODING);
    }

    public static byte[] readFileBinary(File file) throws FileNotFoundException, IOException {
        FileInputStream in = new FileInputStream(file);
        byte[] buffer = new byte[(int)file.length()];
        ByteBuffer byteBuffer = ByteBuffer.wrap(buffer);
        FileChannel channel = in.getChannel();
        try {
            int readSum = 0;
            while (readSum < buffer.length) {
                int read = channel.read(byteBuffer);
                if (read < 0) {
                    throw new IOException("Reached EOF before entire file could be read!");
                }
                readSum += read;
            }
        }
        finally {
            FileSystemUtils.close(channel);
            FileSystemUtils.close(in);
        }
        return buffer;
    }

    public static void unjar(File jarFile, File targetDirectory) throws IOException {
        JarFile jar = new JarFile(jarFile);
        Enumeration<JarEntry> entries = jar.entries();
        while (entries.hasMoreElements()) {
            JarEntry entry = entries.nextElement();
            if (entry.isDirectory()) continue;
            InputStream entryStream = jar.getInputStream(entry);
            File file = new File(targetDirectory, entry.getName());
            FileSystemUtils.ensureParentDirectoryExists(file);
            FileOutputStream outputStream = new FileOutputStream(file);
            FileSystemUtils.copy(entryStream, outputStream);
            entryStream.close();
            outputStream.close();
        }
        jar.close();
    }

    public static void writeFile(File file, String content) throws IOException {
        FileSystemUtils.writeFile(file, content, Charset.defaultCharset().name());
    }

    public static void writeFileUTF8(File file, String content) throws IOException {
        FileSystemUtils.writeFile(file, content, UTF8_ENCODING);
    }

    public static void writeFile(File file, String content, String encoding) throws IOException {
        FileSystemUtils.ensureParentDirectoryExists(file);
        OutputStreamWriter writer = null;
        try {
            writer = new OutputStreamWriter((OutputStream)new FileOutputStream(file), encoding);
            writer.write(content);
        }
        catch (Throwable throwable) {
            FileSystemUtils.close(writer);
            throw throwable;
        }
        FileSystemUtils.close(writer);
    }

    public static void writeFileWithBOM(File file, String content, EByteOrderMark bom) throws IOException {
        FileSystemUtils.ensureParentDirectoryExists(file);
        FileOutputStream out = null;
        try {
            out = new FileOutputStream(file);
            out.write(bom.getBOM());
            OutputStreamWriter writer = new OutputStreamWriter((OutputStream)out, bom.getEncoding());
            writer.write(content);
            writer.flush();
        }
        catch (Throwable throwable) {
            FileSystemUtils.close(out);
            throw throwable;
        }
        FileSystemUtils.close(out);
    }

    private static void listFilesRecursively(File directory, Collection<File> result, FileFilter filter) {
        File[] fileArray = directory.listFiles();
        int n = fileArray.length;
        int n2 = 0;
        while (n2 < n) {
            File f = fileArray[n2];
            if (f.isDirectory()) {
                FileSystemUtils.listFilesRecursively(f, result, filter);
            }
            if (filter == null || filter.accept(f)) {
                result.add(f);
            }
            ++n2;
        }
    }

    public static void mergeTemplate(File templateFile, File outFile, Object ... arguments) throws IOException {
        String output;
        String template = FileSystemUtils.readFile(templateFile);
        try {
            output = String.format(template, arguments);
        }
        catch (IllegalFormatException e) {
            throw new IOException("Illegal format: " + e.getMessage());
        }
        FileSystemUtils.writeFile(outFile, output);
    }

    public static InputStream mergeTemplate(InputStream inStream, Object ... arguments) throws IOException {
        String output;
        String template = FileSystemUtils.readStream(inStream);
        try {
            output = String.format(template, arguments);
        }
        catch (IllegalFormatException e) {
            throw new IOException("Illegal format: " + e.getMessage());
        }
        return new ByteArrayInputStream(output.getBytes());
    }

    public static String readStream(InputStream input) throws IOException {
        return FileSystemUtils.readStream(input, Charset.defaultCharset().name());
    }

    public static String readStreamUTF8(InputStream input) throws IOException {
        return FileSystemUtils.readStream(input, UTF8_ENCODING);
    }

    public static String readStream(InputStream input, String encoding) throws IOException {
        int n;
        StringBuilder out = new StringBuilder();
        Reader r = FileSystemUtils.streamReader(input, encoding);
        char[] b = new char[4096];
        while ((n = r.read(b)) != -1) {
            out.append(b, 0, n);
        }
        return out.toString();
    }

    public static byte[] readStreamBinary(InputStream input) throws IOException {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        FileSystemUtils.copy(input, out);
        return out.toByteArray();
    }

    public static Reader streamReader(InputStream in, String encoding) throws IOException {
        if (!in.markSupported()) {
            in = new BufferedInputStream(in);
        }
        in.mark(4);
        byte[] prefix = new byte[4];
        EByteOrderMark bom = null;
        try {
            FileSystemUtils.safeRead(in, prefix);
            bom = EByteOrderMark.determineBOM(prefix);
        }
        catch (IOException iOException) {
            // empty catch block
        }
        in.reset();
        if (bom != null) {
            encoding = bom.getEncoding();
            int i = 0;
            while (i < bom.getBOMLength()) {
                in.read();
                ++i;
            }
        }
        return new InputStreamReader(in, encoding);
    }

    public static Properties readPropertiesFile(File propertiesFile) throws IOException {
        Properties properties = new Properties();
        FileInputStream inputStream = new FileInputStream(propertiesFile);
        try {
            properties.load(inputStream);
        }
        finally {
            ((InputStream)inputStream).close();
        }
        return properties;
    }

    /*
     * WARNING - void declaration
     */
    public static File commonRoot(Iterable<? extends File> files) throws IOException {
        void var2_7;
        HashSet<String> absolutePaths = new HashSet<String>();
        for (File file : files) {
            absolutePaths.add(file.getCanonicalPath());
        }
        CCSMPre.isTrue(absolutePaths.size() >= 2, "Expected are at least 2 files");
        String string = StringUtils.longestCommonPrefix(absolutePaths);
        int lastSeparator = string.lastIndexOf(File.separator);
        if (lastSeparator > -1) {
            String string2 = string.substring(0, lastSeparator);
        }
        if (StringUtils.isEmpty((String)var2_7)) {
            return null;
        }
        return new File((String)var2_7);
    }

    @Deprecated
    public static String canonize(String path) {
        return FileSystemUtils.canonize(new File(path)).getPath();
    }

    @Deprecated
    public static File canonize(File file) {
        try {
            return file.getCanonicalFile();
        }
        catch (IOException e) {
            throw new AssertionError((Object)("Problems creating canonical path for " + file));
        }
    }

    public static InputStream autoDecompressStream(InputStream in) throws IOException {
        if (!in.markSupported()) {
            in = new BufferedInputStream(in);
        }
        in.mark(2);
        boolean isGZIP = (in.read() & 0xFF | (in.read() & 0xFF) << 8) == 35615;
        in.reset();
        if (isGZIP) {
            return new GZIPInputStream(in);
        }
        return in;
    }

    public static void close(Closeable closeable) {
        FileSystemUtils.close(closeable, null);
    }

    public static void close(Closeable closeable, ILogger logger) {
        block3: {
            if (closeable == null) {
                return;
            }
            try {
                closeable.close();
            }
            catch (IOException e) {
                if (logger == null) break block3;
                logger.warn("Trouble closing: " + e.getMessage());
            }
        }
    }

    public static void sort(List<File> files) {
        Collections.sort(files, new FilenameComparator());
    }

    public static String normalizeSeparators(String path) {
        return path.replace(File.separatorChar, '/');
    }

    public static File extractJarFileFromJarURL(URL url) {
        CCSMPre.isTrue("jar".equals(url.getProtocol()), "May only be used with 'jar' URLs!");
        String path = url.getPath();
        CCSMPre.isTrue(path.startsWith("file:"), "May only be used for URLs pointing to files");
        int index = path.indexOf(33);
        CCSMAssert.isTrue(index >= 0, "Unknown format for jar URLs");
        path = path.substring(0, index);
        return FileSystemUtils.fromURL(path);
    }

    private static File fromURL(String url) {
        url = url.replace(" ", "%20");
        try {
            return new File(new URI(url));
        }
        catch (URISyntaxException e) {
            throw new AssertionError((Object)("The assumption is that this method is capable of working with non-standard-compliant URLs, too. Apparently it is not. Invalid URL: " + url + ". Ex: " + e.getMessage()));
        }
    }

    public static boolean isAbsolutePath(String filename) {
        if (filename.startsWith("/") || filename.startsWith("~/")) {
            return true;
        }
        if (filename.length() > 2 && Character.isLetter(filename.charAt(0)) && filename.charAt(1) == ':') {
            return true;
        }
        return filename.startsWith("\\\\");
    }

    public static void safeRead(InputStream in, byte[] data) throws IOException, EOFException {
        FileSystemUtils.safeRead(in, data, 0, data.length);
    }

    public static void safeRead(InputStream in, byte[] data, int offset, int length) throws IOException, EOFException {
        while (length > 0) {
            int read = in.read(data, offset, length);
            if (read < 0) {
                throw new EOFException("Reached end of file before completing read.");
            }
            offset += read;
            length -= read;
        }
    }

    public static File getTmpDir() {
        return new File(System.getProperty("java.io.tmpdir"));
    }

    public static boolean contentsEqual(File file1, File file2) throws IOException {
        byte[] content1 = FileSystemUtils.readFileBinary(file1);
        byte[] content2 = FileSystemUtils.readFileBinary(file2);
        return Arrays.equals(content1, content2);
    }
}

