/*
 * Decompiled with CFR 0.152.
 */
package org.conqat.lib.commons.filesystem;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.List;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.zip.GZIPOutputStream;
import org.conqat.lib.commons.collections.CollectionUtils;
import org.conqat.lib.commons.filesystem.ClassPathUtils;
import org.conqat.lib.commons.filesystem.EByteOrderMark;
import org.conqat.lib.commons.filesystem.FileExtensionFilter;
import org.conqat.lib.commons.filesystem.FileOnlyFilter;
import org.conqat.lib.commons.filesystem.FileSystemUtils;
import org.conqat.lib.commons.string.StringUtils;
import org.conqat.lib.commons.test.CCSMTestCaseBase;

public class FileSystemUtilsTest
extends CCSMTestCaseBase {
    private static final String FILE_CONTENT = "test_file";
    private static final String[] filenames = new String[]{"file-1.txt", "directory-1/file-1.txt", "directory-1/file-2.txt", "directory-1/directory-1/file1.txt", "directory-1/directory-1/file2.bak", "directory-1/directory-2/file1.txt", "directory-1/directory-2/file2.bak", "directory-1/directory-2/directory-1/file1.txt", "directory-1/directory-2/directory-1/file2.txt", "directory-1/directory-2/directory-1/file3.bak"};

    public void testCopyFile() throws IOException {
        String content = "Test\nString\n";
        File sourceFile = this.createTmpFile("test.txt", content);
        File targetFile = new File(this.getTmpDirectory(), "test2.txt");
        FileSystemUtils.copyFile(sourceFile, targetFile);
        String readContent = FileSystemUtils.readFile(targetFile);
        FileSystemUtilsTest.assertEquals((String)content, (String)readContent);
    }

    public void testCopyFiles() throws IOException {
        File source = this.createDirectoryStructure();
        File target = new File(this.getTmpDirectory(), "target");
        int fileCount = FileSystemUtils.copyFiles(source, target, new FileOnlyFilter());
        FileSystemUtilsTest.assertEquals((int)filenames.length, (int)fileCount);
        String[] stringArray = filenames;
        int n = filenames.length;
        int n2 = 0;
        while (n2 < n) {
            String filename = stringArray[n2];
            File originalFile = new File(source, filename);
            File copiedFile = new File(target, filename);
            FileSystemUtilsTest.assertEquals((long)originalFile.length(), (long)copiedFile.length());
            ++n2;
        }
    }

    public void testCopyStreams() throws IOException {
        String content = StringUtils.randomString(2000);
        File sourceFile = this.createTmpFile("stream_source.txt", content);
        FileInputStream in = new FileInputStream(sourceFile);
        File targetFile = new File(this.getTmpDirectory(), "stream_target.txt");
        FileOutputStream out = new FileOutputStream(targetFile);
        FileSystemUtils.copy(in, out);
        ((InputStream)in).close();
        ((OutputStream)out).close();
        FileSystemUtilsTest.assertEquals((String)content, (String)FileSystemUtils.readFile(targetFile));
    }

    public void testCreateJARFile() throws IOException {
        File source = this.createDirectoryStructure();
        File jarFile = new File("test.jar");
        int fileCount = FileSystemUtils.createJARFile(jarFile, source, new FileOnlyFilter());
        FileSystemUtilsTest.assertEquals((int)filenames.length, (int)fileCount);
        HashSet<String> files = CollectionUtils.asHashSet(filenames);
        JarFile jar = new JarFile(jarFile);
        Enumeration<JarEntry> entries = jar.entries();
        while (entries.hasMoreElements()) {
            JarEntry entry = entries.nextElement();
            FileSystemUtilsTest.assertTrue((boolean)files.contains(entry.getName()));
        }
        jar.close();
        jarFile.delete();
    }

    public void testUnjar() throws IOException {
        File source = this.createDirectoryStructure();
        File jarFile = new File("test.jar");
        FileSystemUtils.createJARFile(jarFile, source, new FileOnlyFilter());
        File targetDirectory = new File(this.getTmpDirectory(), "target");
        FileSystemUtils.unjar(jarFile, targetDirectory);
        List<File> files = FileSystemUtils.listFilesRecursively(targetDirectory, new FileOnlyFilter());
        FileSystemUtilsTest.assertEquals((int)filenames.length, (int)files.size());
        for (File file : files) {
            FileSystemUtilsTest.assertEquals((long)FILE_CONTENT.length(), (long)file.length());
        }
        jarFile.delete();
    }

    public void testCreateRelativePath() throws IOException {
        this.createDirectoryStructure();
        File dir1 = new File(this.getTmpDirectory(), "directory-1");
        File dir1dir1 = new File(this.getTmpDirectory(), "directory-1/directory-1");
        File dir1dir2 = new File(this.getTmpDirectory(), "directory-1/directory-2");
        File dir1dir2dir1 = new File(this.getTmpDirectory(), "directory-1/directory-2/directory-1");
        String path = FileSystemUtils.createRelativePath(dir1, dir1);
        FileSystemUtilsTest.assertEquals((String)"", (String)path);
        path = FileSystemUtils.createRelativePath(dir1, dir1dir1);
        FileSystemUtilsTest.assertEquals((String)"../", (String)path);
        path = FileSystemUtils.createRelativePath(dir1dir1, dir1);
        FileSystemUtilsTest.assertEquals((String)"directory-1/", (String)path);
        path = FileSystemUtils.createRelativePath(dir1dir2dir1, dir1);
        FileSystemUtilsTest.assertEquals((String)"directory-2/directory-1/", (String)path);
        path = FileSystemUtils.createRelativePath(dir1, dir1dir2dir1);
        FileSystemUtilsTest.assertEquals((String)"../../", (String)path);
        path = FileSystemUtils.createRelativePath(dir1dir1, dir1dir2);
        FileSystemUtilsTest.assertEquals((String)"../directory-1/", (String)path);
        path = FileSystemUtils.createRelativePath(dir1dir2, dir1dir1);
        FileSystemUtilsTest.assertEquals((String)"../directory-2/", (String)path);
        path = FileSystemUtils.createRelativePath(dir1dir1, dir1dir2dir1);
        FileSystemUtilsTest.assertEquals((String)"../../directory-1/", (String)path);
        path = FileSystemUtils.createRelativePath(dir1dir2dir1, dir1dir1);
        FileSystemUtilsTest.assertEquals((String)"../directory-2/directory-1/", (String)path);
    }

    public void testDeleteRecursively() throws IOException {
        File root = this.createDirectoryStructure();
        FileSystemUtils.deleteRecursively(root);
        FileSystemUtilsTest.assertFalse((boolean)root.exists());
    }

    public void testEnsureDirectoryExists() throws IOException {
        File file = new File("tmp");
        file.delete();
        FileSystemUtils.ensureDirectoryExists(file);
        FileSystemUtilsTest.assertTrue((boolean)file.exists());
        FileSystemUtilsTest.assertTrue((boolean)file.isDirectory());
        file.delete();
    }

    public void testEnsureParentDirectoryExists() throws IOException {
        File file = new File("tmp" + File.separator + "test.txt");
        file.getParentFile().delete();
        FileSystemUtils.ensureDirectoryExists(file);
        FileSystemUtilsTest.assertTrue((boolean)file.getParentFile().exists());
        FileSystemUtilsTest.assertTrue((boolean)file.getParentFile().isDirectory());
        file.delete();
    }

    public void testEnsureParentDirectoryExistsFileWithoutParent() throws IOException {
        FileSystemUtils.ensureParentDirectoryExists(new File("test"));
    }

    public void testFileExtensionFilter() throws IOException {
        File root = this.createDirectoryStructure();
        List<File> files = FileSystemUtils.listFilesRecursively(root, new FileExtensionFilter("txt", "bak"));
        for (File file : files) {
            FileSystemUtilsTest.assertTrue((file.getAbsolutePath().endsWith(".txt") || file.getAbsolutePath().endsWith(".bak") ? 1 : 0) != 0);
        }
        String[] stringArray = filenames;
        int n = filenames.length;
        int n2 = 0;
        while (n2 < n) {
            String filename = stringArray[n2];
            if (filename.endsWith(".txt") || filename.endsWith("bak")) {
                File file = new File(this.getTmpDirectory(), filename);
                FileSystemUtilsTest.assertTrue((String)("" + file), (boolean)files.contains(file));
            }
            ++n2;
        }
    }

    public void testFileOnlyFilter() throws IOException {
        File root = this.createDirectoryStructure();
        List<File> files = FileSystemUtils.listFilesRecursively(root, new FileOnlyFilter());
        FileSystemUtilsTest.assertEquals((int)files.size(), (int)filenames.length);
        for (File file : files) {
            FileSystemUtilsTest.assertTrue((boolean)file.isFile());
        }
        String[] stringArray = filenames;
        int n = filenames.length;
        int n2 = 0;
        while (n2 < n) {
            String filename = stringArray[n2];
            File file = new File(this.getTmpDirectory(), filename);
            FileSystemUtilsTest.assertTrue((String)("" + file), (boolean)files.contains(file));
            ++n2;
        }
    }

    public void testListRecursively() throws IOException {
        File root = this.createDirectoryStructure();
        List<File> files = FileSystemUtils.listFilesRecursively(root);
        String[] stringArray = filenames;
        int n = filenames.length;
        int n2 = 0;
        while (n2 < n) {
            String filename = stringArray[n2];
            File file = new File(this.getTmpDirectory(), filename);
            FileSystemUtilsTest.assertTrue((boolean)files.contains(file));
            File directory = file.getParentFile();
            if (!directory.equals(this.getTmpDirectory())) {
                FileSystemUtilsTest.assertTrue((String)(directory + " ist not in list"), (boolean)files.contains(directory));
            }
            ++n2;
        }
    }

    public void testNewFile() {
        File parent = new File("parent");
        File newFile = FileSystemUtils.newFile(parent, new String[0]);
        FileSystemUtilsTest.assertEquals((Object)parent, (Object)newFile);
        newFile = FileSystemUtils.newFile(parent, "child1");
        FileSystemUtilsTest.assertEquals((String)("parent" + File.separator + "child1"), (String)newFile.getPath());
        newFile = FileSystemUtils.newFile(parent, "child1", "child2");
        FileSystemUtilsTest.assertEquals((String)("parent" + File.separator + "child1" + File.separator + "child2"), (String)newFile.getPath());
        newFile = FileSystemUtils.newFile(parent, "child1", "child2", "child3");
        FileSystemUtilsTest.assertEquals((String)("parent" + File.separator + "child1" + File.separator + "child2" + File.separator + "child3"), (String)newFile.getPath());
    }

    public void testReadWriteFile() throws IOException {
        String content = "Test\nString\n";
        File file = new File(this.getTmpDirectory() + "test.txt");
        FileSystemUtils.writeFile(file, content);
        String readContent = FileSystemUtils.readFile(file);
        FileSystemUtilsTest.assertEquals((String)content, (String)readContent);
    }

    public void testReadFileUTF() throws IOException {
        File file = new File(this.getTmpDirectory() + "test.txt");
        String content = "Test\nString\n\u00c4\u00d6\u00dc\u00e4\u00f6\u00fc";
        EByteOrderMark[] eByteOrderMarkArray = EByteOrderMark.values();
        int n = eByteOrderMarkArray.length;
        int n2 = 0;
        while (n2 < n) {
            EByteOrderMark bom = eByteOrderMarkArray[n2];
            FileSystemUtils.writeFileWithBOM(file, "Test\nString\n\u00c4\u00d6\u00dc\u00e4\u00f6\u00fc", bom);
            String read = FileSystemUtils.readFile(file, Charset.defaultCharset().name());
            FileSystemUtilsTest.assertEquals((String)("Expected correct content for encoding " + bom.getEncoding()), (String)"Test\nString\n\u00c4\u00d6\u00dc\u00e4\u00f6\u00fc", (String)read);
            ++n2;
        }
    }

    private File createDirectoryStructure() throws IOException {
        this.deleteTmpDirectory();
        String[] stringArray = filenames;
        int n = filenames.length;
        int n2 = 0;
        while (n2 < n) {
            String filename = stringArray[n2];
            this.createTmpFile(filename, FILE_CONTENT);
            ++n2;
        }
        return this.getTmpDirectory();
    }

    public void testGetFileExtension() {
        FileSystemUtilsTest.assertEquals((String)"a", (String)this.extension("name.a"));
        FileSystemUtilsTest.assertEquals((String)"ab", (String)this.extension("name.ab"));
        FileSystemUtilsTest.assertEquals((String)"abc", (String)this.extension("othername.abc"));
        FileSystemUtilsTest.assertEquals((String)"java", (String)this.extension("name with spaces.java"));
        FileSystemUtilsTest.assertEquals((String)"c", (String)this.extension("a.b.c"));
        FileSystemUtilsTest.assertEquals((String)"cs", (String)this.extension("c:/windows/system21/somefile.cs"));
        FileSystemUtilsTest.assertEquals(null, (String)this.extension("name_without_extension"));
        FileSystemUtilsTest.assertEquals(null, (String)this.extension(""));
        FileSystemUtilsTest.assertEquals((String)"", (String)this.extension("somename."));
    }

    private String extension(String filename) {
        return FileSystemUtils.getFileExtension(new File(filename));
    }

    public void testMergeTemplateError() {
        File templateFile = this.useTestFile("template01.txt");
        File outFile = new File(this.getTmpDirectory(), "out1.txt");
        try {
            FileSystemUtils.mergeTemplate(templateFile, outFile, new Object[0]);
            FileSystemUtilsTest.fail();
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    public void testMergeTemplate() throws IOException {
        File templateFile = this.useTestFile("template01.txt");
        File outFile = new File(this.getTmpDirectory(), "out1.txt");
        FileSystemUtils.mergeTemplate(templateFile, outFile, "arg1");
        FileSystemUtilsTest.assertEquals((String)"test template: arg1", (String)FileSystemUtils.readFile(outFile));
    }

    public void testReadStream() throws IOException {
        FileInputStream stream = new FileInputStream(this.useTestFile("input01.txt"));
        FileSystemUtilsTest.assertEquals((String)("1234567890" + StringUtils.CR + "1234567890"), (String)FileSystemUtils.readStream(stream));
        stream.close();
    }

    public void testMergeTemplateStreams() throws IOException {
        FileInputStream inStream = new FileInputStream(this.useTestFile("template01.txt"));
        InputStream stream = FileSystemUtils.mergeTemplate(inStream, "arg1");
        FileSystemUtilsTest.assertEquals((String)"test template: arg1", (String)FileSystemUtils.readStream(stream));
        ((InputStream)inStream).close();
        stream.close();
    }

    public void testCommonRootDifferentPathStyles() {
        String root = FileSystemUtilsTest.determineRoot();
        this.assertCommonRoot(new String[]{String.valueOf(root.toLowerCase()) + "root/a.txt", String.valueOf(root) + "root" + File.separator + "b.txt", String.valueOf(root.toUpperCase()) + "root/c.txt"}, new File(String.valueOf(root) + "root/"));
    }

    private static String determineRoot() {
        return File.listRoots()[0].getPath();
    }

    public void testCommonRootNeighbouringFiles() {
        String root = String.valueOf(FileSystemUtilsTest.determineRoot()) + "root/";
        this.assertCommonRoot(new String[]{String.valueOf(root) + "a.txt", String.valueOf(root) + "b.txt", String.valueOf(root) + "c.txt"}, new File(root));
    }

    public void testCommonRootFilesWithEqualPrefix() {
        String root = String.valueOf(FileSystemUtilsTest.determineRoot()) + "root/";
        this.assertCommonRoot(new String[]{String.valueOf(root) + "PREa.txt", String.valueOf(root) + "PREb.txt"}, new File(root));
    }

    public void testCommonRootNoRoot() {
        if (System.getProperty("os.name").startsWith("Windows")) {
            this.assertCommonRoot(new String[]{"a:/root1/a.txt", "b:/root2/b.txt"}, null);
        }
    }

    private void assertCommonRoot(String[] paths, File expectedRoot) {
        ArrayList<File> files = new ArrayList<File>();
        String[] stringArray = paths;
        int n = paths.length;
        int n2 = 0;
        while (n2 < n) {
            String path = stringArray[n2];
            files.add(new File(path));
            ++n2;
        }
        File actualRoot = null;
        try {
            actualRoot = FileSystemUtils.commonRoot(files);
        }
        catch (IOException e) {
            FileSystemUtilsTest.fail((String)("IOException thrown: " + e.getMessage()));
        }
        FileSystemUtilsTest.assertEquals((Object)expectedRoot, (Object)actualRoot);
    }

    public void testAutoDecompressStream() throws IOException {
        byte[] data = "Some redundant text to actually allow compression, redundant, compression!".getBytes();
        byte[] result = new byte[2 * data.length];
        ByteArrayInputStream bis = new ByteArrayInputStream(data);
        int read = FileSystemUtils.autoDecompressStream(bis).read(result);
        FileSystemUtilsTest.assertEquals((int)data.length, (int)read);
        int i = 0;
        while (i < data.length) {
            FileSystemUtilsTest.assertEquals((byte)data[i], (byte)result[i]);
            ++i;
        }
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        GZIPOutputStream out = new GZIPOutputStream(bos);
        out.write(data);
        out.close();
        bis = new ByteArrayInputStream(bos.toByteArray());
        read = FileSystemUtils.autoDecompressStream(bis).read(result);
        FileSystemUtilsTest.assertEquals((int)data.length, (int)read);
        int i2 = 0;
        while (i2 < data.length) {
            FileSystemUtilsTest.assertEquals((byte)data[i2], (byte)result[i2]);
            ++i2;
        }
    }

    public void testExtractJarFileFromJarURL() throws IOException, URISyntaxException {
        URL jreURL = ClassPathUtils.obtainClassFileURL(Integer.class);
        FileSystemUtilsTest.assertEquals((String)"jar", (String)jreURL.getProtocol());
        File jarFile = FileSystemUtils.extractJarFileFromJarURL(jreURL);
        FileSystemUtilsTest.assertTrue((String)"JDK JARs should exist!", (boolean)jarFile.isFile());
        FileSystemUtilsTest.assertTrue((boolean)jarFile.getPath().toLowerCase().endsWith(".jar"));
        File file = this.useTestFile("directory with spaces/test.jar");
        URL url = new URL("jar:" + file.toURI().toURL() + "!/canonical.txt");
        jarFile = FileSystemUtils.extractJarFileFromJarURL(url);
        FileSystemUtilsTest.assertTrue((String)"File should exist!", (boolean)jarFile.isFile());
        file = new File("C:/Dokumente und Einstellungen/lochmann/Eigene Dateien/Downloads/conqat-quamoco-win32(4)/conqat-2.7/dropins");
        url = new URL("jar:" + file.toURI().toURL() + "!/dummy.txt");
        FileSystemUtils.extractJarFileFromJarURL(url);
        url = new URL("jar:file:/C:/Dokumente und Einstellungen/lochmann/Eigene Dateien/Downloads/conqat-quamoco-win32(4)/conqat-2.7/dropins!/dummy.txt");
        FileSystemUtils.extractJarFileFromJarURL(url);
    }

    public void testIsAbsolutePath() {
        String[] relativePaths;
        String[] absolutePaths;
        String[] stringArray = absolutePaths = new String[]{"c:\\a.txt", "C:\\b.txt", "f:/a/b.txt", "Y:\\a", "C:\\", "z:/", "\\\\smbbroy\\juergens", "~/usr/html.txt", "/usr/trala/a.txt", "/usr/trala/a.txt"};
        int n = absolutePaths.length;
        int n2 = 0;
        while (n2 < n) {
            String absolutePath = stringArray[n2];
            FileSystemUtilsTest.assertTrue((String)("Path '" + absolutePath + "' is considered not absolute, but should be!"), (boolean)FileSystemUtils.isAbsolutePath(absolutePath));
            ++n2;
        }
        String[] stringArray2 = relativePaths = new String[]{"a", "a.txt", "a/b/c", "a\\b\\c", "a\\b.txt"};
        int n3 = relativePaths.length;
        n = 0;
        while (n < n3) {
            String relativePath = stringArray2[n];
            FileSystemUtilsTest.assertFalse((String)("Path '" + relativePath + "' is considered not relative, but should be!"), (boolean)FileSystemUtils.isAbsolutePath(relativePath));
            ++n;
        }
    }

    public void testSafeReadHappyCase() throws Exception {
        byte[] sampleData = new byte[]{1, 2, 3};
        ByteArrayInputStream bis = new ByteArrayInputStream(sampleData);
        byte[] buf = new byte[3];
        FileSystemUtils.safeRead(bis, buf);
        FileSystemUtilsTest.assertTrue((boolean)Arrays.equals(sampleData, buf));
    }

    public void testSafeReadExceptionCase() throws Exception {
        byte[] sampleData = new byte[]{1, 2, 3};
        ByteArrayInputStream bis = new ByteArrayInputStream(sampleData);
        try {
            FileSystemUtils.safeRead(bis, new byte[4]);
            FileSystemUtilsTest.fail((String)"EOFException expected");
        }
        catch (EOFException eOFException) {
            // empty catch block
        }
    }

    public void testSafeReadMultipleReads() throws Exception {
        ByteArrayInputStream bis = new ByteArrayInputStream(new byte[0]){
            private boolean firstCall;
            {
                this.firstCall = true;
            }

            @Override
            public synchronized int read(byte[] b, int off, int len) {
                if (this.firstCall) {
                    b[0] = 1;
                    this.firstCall = false;
                    return 1;
                }
                b[1] = 2;
                b[2] = 3;
                return 2;
            }
        };
        byte[] buf = new byte[3];
        FileSystemUtils.safeRead(bis, buf);
        FileSystemUtilsTest.assertTrue((boolean)Arrays.equals(new byte[]{1, 2, 3}, buf));
    }
}

